<?php

namespace uncanny_automator;


/**
 * Class Automator_Url_Validation
 * @package uncanny_automator
 */
class Automator_Input_Parser {

	public $defined_tokens = [
		'site_name',
		'user_username',
		'user_firstname',
		'user_lastname',
		'user_email',
		'user_displayname',
                'reset_pass_link',
		'admin_email',
		'site_url',
		'recipe_name',
	];

	public $url_regx = "/(?:(?:https?|ftp):\/\/)(?:\S+(?::\S*)?@)?(?:(?!(?:10|127)(?:\.\d{1,3}){3})(?!(?:169\.254|192\.168)(?:\.\d{1,3}){2})(?!172\.(?:1[6-9]|2\d|3[0-1])(?:\.\d{1,3}){2})(?:[1-9]\d?|1\d\d|2[01]\d|22[0-3])(?:\.(?:1?\d{1,2}|2[0-4]\d|25[0-5])){2}(?:\.(?:[1-9]\d?|1\d\d|2[0-4]\d|25[0-4]))|(?:(?:[a-z\x{00a1}-\x{ffff}0-9]-*)*[a-z\x{00a1}-\x{ffff}0-9]+)(?:\.(?:[a-z\x{00a1}-\x{ffff}0-9]-*)*[a-z\x{00a1}-\x{ffff}0-9]+)*(?:\.(?:[a-z\x{00a1}-\x{ffff}]{2,}))\.?)(?::\d{2,5})?(?:[\/?#]\S*)?$/u";

	public function __construct() {
	}

	/**
	 * @param null $url
	 * @param null $recipe_id
	 * @param $trigger_args
	 *
	 * @return mixed|null|string
	 */
	public function url( $url = null, $recipe_id = null, $trigger_args = [] ) {

		// Sanity check that there was a $field_text passed
		if ( null === $url ) {
			#Utilities::log( 'ERROR: You are try to parse an url without passing $url.', 'parse->url ERROR', false, 'uap - errors' );

			return null;
		}

		$original_url = $url;

		if ( '/' === (string) $url[0] ) {
			// Relative URL prefix with site url
			$url = '{{site_url}}' . $url;
		}


		// Replace Tokens
		$args = [
			'field_text'  => $url,
			'meta_key'    => null,
			'user_id'     => null,
			'action_data' => null,
			'recipe_id'   => $recipe_id,
		];

		$url = $this->parse_vars( $args, $trigger_args );

		if ( ! preg_match( $this->url_regx, $url ) ) {
			// if the url is not valid and / isn't the first character then the url is missing the site url
			$url = '{{site_url}}' . '/' . $url;
			// Replace Tokens
			$args = [
				'field_text'  => $url,
				'meta_key'    => null,
				'user_id'     => null,
				'action_data' => null,
				'recipe_id'   => $recipe_id,
			];
			$url  = $this->parse_vars( $args, $trigger_args );

		}

		// Replace all spaces with %20
		$url = str_replace( ' ', '%20', $url );

		if ( ! preg_match( $this->url_regx, $url ) ) {
			// if the url is not valid still then something when wrong...
			#Utilities::log( 'ERROR: The URL passed could not be parsed properly. @$original_url: ' . $original_url . '@failed Url: ' . $url, 'parse->url ERROR', false, 'uap - errors' );

			return null;
		}

		return $url;
	}

	/**
	 * @param null $field_text
	 * @param null $recipe_id
	 * @param null $user_id
	 *
	 * @param null $trigger_args
	 *
	 * @return null|string
	 */
	public function text( $field_text = null, $recipe_id = null, $user_id = null, $trigger_args = null ) {

		// Sanity check that there was a $field_text passed
		if ( null === $field_text ) {
			#Utilities::log( 'ERROR: You are try to parse a field text\'s token without passing $field_text.', 'parse->text ERROR', false, 'uap - errors' );

			return null;
		}
		$args = [
			'field_text'     => $field_text,
			'meta_key'       => null,
			'user_id'        => $user_id,
			'action_data'    => null,
			'recipe_id'      => $recipe_id,
			'trigger_log_id' => $trigger_args['get_trigger_id'],
			'run_number'     => $trigger_args['run_number'],
		];

		#Utilities::log( $args, '$args-later', true, 'args-zapier' );

		return $this->parse_vars( $args, $trigger_args );
	}


	/**
	 * Parse field text by replacing variable with real data
	 *
	 * @param $args
	 * @param $trigger_args
	 *
	 * @return string
	 */
	public function parse_vars( $args, $trigger_args = [] ) {
		$field_text     = $args['field_text'];
		$meta_key       = $args['meta_key'];
		$user_id        = $args['user_id'];
		$action_data    = $args['action_data'];
		$recipe_id      = $args['recipe_id'];
		$trigger_log_id = key_exists( 'trigger_log_id', $args ) ? $args['trigger_log_id'] : null;
		$run_number     = key_exists( 'run_number', $args ) ? $args['run_number'] : null;

		// find brackets and replace with real data
		if ( preg_match_all( "/\{\{\s*(.*?)\s*\}\}/", $field_text, $arr ) ) {

			$matches = $arr[1];
			foreach ( $matches as $match ) {

				$replaceable = '';

				if ( false !== strpos( $match, ':' ) ) {
					if ( preg_match( '/(USERMETA)/', $match ) ) {
						//Usermeta found!!
						if ( is_null( $user_id ) && 0 !== absint( $user_id ) ) {
							$user_id = wp_get_current_user()->ID;
						}
						if ( 0 !== $user_id ) {
							$pieces = explode( ':', $match );
							switch ( $pieces[0] ) {
								case 'USERMETAEMAIL':
									$user_meta = get_user_meta( $user_id, $pieces[1], true );

									$replaceable = is_email( $user_meta ) ? $user_meta : '';
									break;
								case 'USERMETA':
									$user_data = get_userdata( $user_id );
									$user_data = (array) $user_data->data;
									if ( isset( $user_data[ $pieces[1] ] ) ) {
										$replaceable = $user_data[ $pieces[1] ];
									} else {
										$user_meta   = get_user_meta( $user_id, $pieces[1], true );
										$replaceable = $user_meta;
									}
									break;
							}
						}
						$field_text = str_replace( $match, $replaceable, $field_text );
					} else {
						//Non usermeta
						global $wpdb;
						$qq          = "SELECT meta_value FROM {$wpdb->prefix}uap_trigger_log_meta WHERE meta_key = 'parsed_data' AND automator_trigger_log_id = $trigger_log_id AND user_id = $user_id AND run_number = $run_number ";
						$parsed_data = $wpdb->get_var( $qq );
						$run_func    = true;
						/*Utilities::log( [
							'$qq' => $qq,
						], '', true, 'parsed' );*/
						if ( ! empty( $parsed_data ) ) {
							$parsed_data = maybe_unserialize( $parsed_data );
							if ( key_exists( '{{' . $match . '}}', $parsed_data ) && ! empty( $parsed_data[ '{{' . $match . '}}' ] ) ) {
								$replaceable = $parsed_data[ '{{' . $match . '}}' ];
								$run_func    = false;
							} else {
								$run_func = true;
							}
							/*Utilities::log( [
								'$replaceable' => $replaceable,
								'$parsed_data' => $parsed_data,
								'$match'       => $match
							], '', true, 'parsed' );*/
						}
						if ( empty( $replaceable ) ) {
							$run_func = true;
						}
						if ( $run_func ) {
							$pieces = explode( ':', $match );
							if ( $pieces ) {
								$replace_args = [
									'pieces'         => $pieces,
									'recipe_id'      => $recipe_id,
									'trigger_log_id' => $trigger_log_id,
									'run_number'     => $run_number,
									'user_id'        => $user_id,
								];
								/*Utilities::log( [
									'$replace_args' => $replace_args,
									'$trigger_args' => $trigger_args,
								], '', true, 'parsed' );*/
								$replaceable = $this->replace_recipe_variables( $replace_args, $trigger_args );
							}
						}
					}
				} elseif ( in_array( $match, $this->defined_tokens ) ) {
					if ( null === $user_id ) {
						$current_user = wp_get_current_user();
					} else {
						$current_user = get_user_by( 'ID', $user_id );
					}

					switch ( $match ) {
						case 'site_name':
							$replaceable = get_bloginfo( 'name' );
							break;

						case 'user_username':
							$replaceable = $current_user->user_login;
							break;

						case 'user_firstname':
							$replaceable = $current_user->user_firstname;
							break;

						case 'user_lastname':
							$replaceable = $current_user->user_lastname;
							break;

						case 'user_email':
							$replaceable = $current_user->user_email;
							break;

						case 'user_displayname':
							$replaceable = $current_user->display_name;
							break;
                                                    
                                                case 'reset_pass_link':
							$replaceable = $this->generate_reset_token( $user_id );
							break;

						case 'admin_email':
							$replaceable = get_bloginfo( 'admin_email' );
							break;

						case 'site_url':
							$replaceable = get_site_url();
							break;

						case 'recipe_name':
							$recipe = get_post( $recipe_id );
							if ( null !== $recipe ) {
								$replaceable = $recipe->post_title;
							}
							break;
					}
				}

				$field_text = str_replace( $match, $replaceable, $field_text );
			}
		}

		$field_text = str_replace( [ '{{', '}}' ], '', $field_text );

		return $field_text;

	}

	/**
	 * @param $replace_args
	 * @param $args
	 *
	 * @return string
	 */
	public function replace_recipe_variables( $replace_args, $args = [] ) {
		global $uncanny_automator;
		$pieces         = $replace_args['pieces'];
		$recipe_id      = $replace_args['recipe_id'];
		$trigger_log_id = $replace_args['trigger_log_id'];
		$run_number     = $replace_args['run_number'];
		$user_id        = $replace_args['user_id'];
		$trigger_data   = $uncanny_automator->get_recipe_data( 'uo-trigger', $recipe_id );
		$return         = '';
		$trigger_id     = absint( $pieces[0] );

		if ( is_null( $user_id ) && 0 !== absint( $user_id ) ) {
			$user_id = wp_get_current_user()->ID;
		}

		foreach ( $trigger_data as $trigger ) {
			foreach ( $pieces as $piece ) {
				if ( key_exists( $piece, $trigger['meta'] ) ) {
					if ( is_numeric( $trigger['meta'][ $piece ] ) ) {
						switch ( $piece ) {
							case 'WPPOST':
							case 'WPPAGE':
								$return = get_the_title( $trigger['meta'][ $piece ] );
								break;
							case 'NUMTIMES':
								$return = $trigger['meta'][ $piece ];
								break;
							case 'WPUSER':
								$user_id = absint( $trigger['meta'][ $piece ] );
								$user    = get_user_by( 'ID', $user_id );
								if ( $user ) {
									return $user->user_email;
								} else {
									return '';
								}
								break;
							default:
								if ( intval( '-1' ) === intval( $trigger['meta'][ $piece ] ) ) {
									//Find stored post_id for piece, i.e., LDLESSON, LDTOPIC set to Any
									$post_id = $uncanny_automator->get->maybe_get_meta_value_from_trigger_log( $piece, $trigger_id, $trigger_log_id, $run_number, $user_id );
									if ( is_numeric( $post_id ) ) {
										$return = get_the_title( $post_id );
									} else {
										$return = __( 'Any', 'uncanny-automator' );
									}
								} elseif ( ! preg_match( '/ANON/', $piece ) ) {
									$return = get_the_title( $trigger['meta'][ $piece ] );
								} else {
									$return = '';
								}
								break;
						}
					} else {
						//Non numeric data.. passed custom post type
						switch ( $piece ) {
							case 'WPPOSTTYPES':
								$return = key_exists( 'post_type_label', $args ) && ! empty( $args['post_type_label'] ) ? $args['post_type_label'] : '';
								break;
						}
					}
				}
			}
		}

		return apply_filters( 'automator_maybe_parse_token', $return, $pieces, $recipe_id, $trigger_data, $user_id );
	}

        /**
	 * @param $user_id
	 *
	 * @return bool|string
	 */
	public function generate_reset_token( $user_id ) {

                $user       = get_user_by( 'ID', $user_id );
                $adt_rp_key = get_password_reset_key( $user );
                $user_login = $user->user_login;
                $url        = network_site_url( "wp-login.php?action=rp&key=$adt_rp_key&login=" . rawurlencode( $user_login ), 'login' );
                $text       = __( 'Click here to reset password.', 'uncanny-automator' );
                $rp_link    = sprintf( '<a href="%s">%s</a>', $url, $text );

                return $rp_link;

	}

}