<?php

namespace Uncanny_Automator_Pro;

/**
 * Class CF7_SUBFIELD
 * @package Uncanny_Automator_Pro
 */
class CF7_SUBFIELD {

	/**
	 * Integration code
	 * @var string
	 */
	public static $integration = 'CF7';

	private $trigger_code;
	private $trigger_meta;
	private $tag_types;

	/**
	 * Set up Automator trigger constructor.
	 */
	public function __construct() {
		$this->trigger_code = 'CF7SUBFIELD';
		$this->trigger_meta = 'CF7FIELDS';
		add_filter( 'wpcf7_verify_nonce', '__return_true' );
		$this->define_trigger();
		add_action( 'wp_ajax_select_form_fields_CF7FORMS', array( $this, 'select_form_fields_func' ) );
	}

	/**
	 * Define and register the trigger by pushing it into the Automator object
	 */
	public function define_trigger() {
		global $uncanny_automator;

		$trigger = array(
			'author'              => $uncanny_automator->get_author_name( $this->trigger_code ),
			'support_link'        => $uncanny_automator->get_author_support_link( $this->trigger_code ),
			'integration'         => self::$integration,
			'code'                => $this->trigger_code,
			/* Translators: 1:Forms*/
			'sentence'            => sprintf( __( 'User submits {{a form:%1$s}} with {{a specific value:%2$s}} in {{a specific field:%3$s}}', 'uncanny-automator' ), $this->trigger_meta, 'SUBVALUE', $this->trigger_code ),
			'select_option_name'  => __( 'User submits {{a form}} with {{a specific value}} in {{a specific field}}', 'uncanny-automator' ),
			'action'              => 'wpcf7_submit',
			'priority'            => 99,
			'accepted_args'       => 2,
			'validation_function' => array( $this, 'wpcf7_submit' ),
			'options_group'       => [
				$this->trigger_meta => [
					$uncanny_automator->options->list_contact_form7_forms( null, $this->trigger_meta, [
						'token'        => false,
						'is_ajax'      => true,
						'target_field' => $this->trigger_code,
						'endpoint'     => 'select_form_fields_CF7FORMS',
					] ),
					$uncanny_automator->options->select_field( $this->trigger_code, __( 'Select a field', 'uncanny-automator' ) ),
					$uncanny_automator->options->text_field( 'SUBVALUE' ),
				],
			],
		);

		$uncanny_automator->register_trigger( $trigger );
	}

	/**
	 * Validation function when the trigger action is hit
	 *
	 * @param $form
	 * @param $result
	 */
	public function wpcf7_submit( $form, $result ) {

		if ( 'validation_failed' !== $result['status'] ) {

			global $uncanny_automator;

			$user_id = get_current_user_id();

			if ( empty( $form ) ) {
				return;
			}

			$recipes    = $uncanny_automator->get->recipes_from_trigger_code( $this->trigger_code );
			$conditions = $this->match_condition( $form, $recipes, $this->trigger_meta, $this->trigger_code, 'SUBVALUE' );

			if ( ! $conditions ) {
				return;
			}
			do_action( 'automator_save_cf7_form', $form, $recipes );
			if ( ! empty( $conditions ) ) {
				foreach ( $conditions['recipe_ids'] as $recipe_id ) {
					if ( ! $uncanny_automator->is_recipe_completed( $recipe_id, $user_id ) ) {
						$args = [
							'code'            => $this->trigger_code,
							'meta'            => $this->trigger_meta,
							'recipe_to_match' => $recipe_id,
							'ignore_post_id'  => true,
							'user_id'         => $user_id,
						];

						$uncanny_automator->maybe_add_trigger_entry( $args );
					}
				}
			}
		}
	}

	/**
	 * @param      $form
	 * @param null $recipes
	 * @param null $trigger_meta
	 * @param null $trigger_code
	 * @param null $trigger_second_code
	 *
	 * @return array|bool
	 */
	public function match_condition( $form, $recipes = null, $trigger_meta = null, $trigger_code = null, $trigger_second_code = null ) {

		if ( null === $recipes ) {
			return false;
		}

		$matches        = [];
		$recipe_ids     = [];
		$entry_to_match = $form->id();

		foreach ( $recipes as $recipe ) {
			foreach ( $recipe['triggers'] as $trigger ) {
				if ( key_exists( $trigger_meta, $trigger['meta'] ) && (string) $trigger['meta'][ $trigger_meta ] === (string) $entry_to_match ) {
					$matches[ $recipe['ID'] ]    = [
						'field' => $trigger['meta'][ $trigger_code ],
						'value' => $trigger['meta'][ $trigger_second_code ],
					];
					$recipe_ids[ $recipe['ID'] ] = $recipe['ID'];
					break;
				}
			}
		}

		if ( ! empty( $matches ) ) {
			foreach ( $matches as $recipe_id => $match ) {
				if ( $_POST[ $match['field'] ] != $match['value'] ) {
					unset( $recipe_ids[ $recipe_id ] );
				}
			}
		}

		if ( ! empty( $recipe_ids ) ) {
			return [ 'recipe_ids' => $recipe_ids, 'result' => true ];
		}

		return false;
	}

	/**
	 * Return all the specific fields of a form ID provided in ajax call
	 */
	public function select_form_fields_func() {
		global $uncanny_automator;

		$uncanny_automator->utilities->ajax_auth_check( $_POST );

		$fields = [];

		if ( isset( $_POST ) ) {
			$form_id     = $_POST['value'];
			$integration = $_POST['integration'];

			$this->load_field_types();

			foreach ( $this->tag_types as $key => $tag ) {
				wpcf7_add_form_tag( $key, $tag['function'], $tag['features'] );
			}

			$wpcform   = \WPCF7_ContactForm::get_instance( $form_id );
			if( !empty($wpcform)) {
				$form_tags = $wpcform->scan_form_tags();
				
				if ( ! empty( $form_tags ) ) {
					foreach ( $form_tags as $tag ) {
						if ( $tag->type !== 'submit' ) {
							$fields[] = [
								'value' => $tag->name,
								'text'  => $tag->name . ' [' . $tag->type . ']',
							];
						}
					}
				}
			}
		}

		echo wp_json_encode( $fields );
		die();
	}

	/**
	 * A list of available tags of contact form 7
	 */
	private function load_field_types() {
		$this->tag_types = array(
			'acceptance' => array(
				'function' => 'wpcf7_acceptance_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'checkbox'   => array(
				'function' => 'wpcf7_checkbox_form_tag_handler',
				'features' => array(
					'name-attr'                   => 1,
					'selectable-values'           => 1,
					'multiple-controls-container' => 1,
				),
			),
			'checkbox*'  => array(
				'function' => 'wpcf7_checkbox_form_tag_handler',
				'features' => array(
					'name-attr'                   => 1,
					'selectable-values'           => 1,
					'multiple-controls-container' => 1,
				),
			),
			'radio'      => array(
				'function' => 'wpcf7_checkbox_form_tag_handler',
				'features' => array(
					'name-attr'                   => 1,
					'selectable-values'           => 1,
					'multiple-controls-container' => 1,
				),
			),
			'count'      => array(
				'function' => 'wpcf7_count_form_tag_handler',
				'features' => array(
					'name-attr'               => 1,
					'zero-controls-container' => 1,
					'not-for-mail'            => 1,
				),
			),
			'date'       => array(
				'function' => 'wpcf7_date_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'date*'      => array(
				'function' => 'wpcf7_date_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'file'       => array(
				'function' => 'wpcf7_file_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'file*'      => array(
				'function' => 'wpcf7_file_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'number'     => array(
				'function' => 'wpcf7_number_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'number*'    => array(
				'function' => 'wpcf7_number_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'range'      => array(
				'function' => 'wpcf7_number_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'range*'     => array(
				'function' => 'wpcf7_number_form_tag_handler',
				'features' => array( 'name-attr' => 1 ),
			),
			'quiz'       => array(
				'function' => 'wpcf7_quiz_form_tag_handler',
				'features' => array(
					'name-attr'    => 1,
					'do-not-store' => 1,
					'not-for-mail' => 1,
				),
			),
			'captchac'   => array(
				'function' => 'wpcf7_captchac_form_tag_handler',
				'features' => array(
					'name-attr'               => 1,
					'zero-controls-container' => 1,
					'not-for-mail'            => 1,
				),
			),
			'captchar'   => array(
				'function' => 'wpcf7_captchar_form_tag_handler',
				'features' => array(
					'name-attr'    => 1,
					'do-not-store' => 1,
					'not-for-mail' => 1,
				),
			),
			'response'   => array(
				'function' => 'wpcf7_response_form_tag_handler',
				'features' => array(
					'display-block' => 1,
				),
			),
			'select'     => array(
				'function' => 'wpcf7_select_form_tag_handler',
				'features' => array(
					'name-attr'         => 1,
					'selectable-values' => 1,
				),
			),
			'select*'    => array(
				'function' => 'wpcf7_select_form_tag_handler',
				'features' => array( 'name-attr' => 1, 'selectable-values' => 1 )
			),
			'submit'     => array( 'function' => 'wpcf7_submit_form_tag_handler', 'features' => array() ),
			'text'       => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'text*'      => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'email'      => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'email*'     => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'url'        => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'url*'       => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'tel'        => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'tel*'       => array(
				'function' => 'wpcf7_text_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'textarea'   => array(
				'function' => 'wpcf7_textarea_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'textarea*'  => array(
				'function' => 'wpcf7_textarea_form_tag_handler',
				'features' => array( 'name-attr' => 1 )
			),
			'hidden'     => array(
				'function' => 'wpcf7_hidden_form_tag_handler',
				'features' => array( 'name-attr' => 1, 'display-hidden' => 1 )
			)
		);
	}
}