<?php

namespace Uncanny_Automator_Pro;

/**
 * Class Internal_Triggers_Actions
 * @package uncanny_automator
 */
class Internal_Triggers_Actions {
	/**
	 * The directories that are auto loaded and initialized
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array
	 */
	private $auto_loaded_directories = null;


	/**
	 * constructor.
	 */
	public function __construct() {

		$directory    = dirname( InitializePlugin::MAIN_FILE ) . '/src/integrations';
		$integrations = glob( $directory . '/*', GLOB_ONLYDIR );

		$this->auto_loaded_directories = $integrations;

		add_action( 'uncanny_automator_add_integration', array( $this, 'init' ), 11 );
		add_action( 'uncanny_automator_add_integration_triggers_actions_tokens', array( $this, 'boot_triggers_actions_closures' ), 15 );

		spl_autoload_register( array( $this, 'require_triggers_actions' ) );
	}

	public function init() {
		$this->initialize_add_integrations();
		$this->auto_loaded_directories = apply_filters( 'uncanny_automator_pro_integration_directory', $this->auto_loaded_directories );
		do_action( 'uncanny_automator_pro_loaded' );
	}

	public function initialize_add_integrations() {

		// Check each directory
		if ( $this->auto_loaded_directories ) {

			foreach ( $this->auto_loaded_directories as $directory ) {
				$files = array_diff( scandir( $directory ), array(
					'..',
					'.',
					'index.php',
					'actions',
					'triggers',
					'tokens',
					'closures',
				) );

				if ( $files ) {
					foreach ( $files as $file ) {
						$file = "$directory/$file";
						if ( file_exists( $file ) ) {
							$file_name = basename( $file, '.php' );

							// Split file name on - eg: my-class-name to array( 'my', 'class', 'name')
							$class_to_filename = explode( '-', $file_name );

							// Make the first letter of each word in array upper case - eg array( 'my', 'class', 'name') to array( 'My', 'Class', 'Name')
							$class_to_filename = array_map( function ( $word ) {
								return ucfirst( $word );
							}, $class_to_filename );

							// Implode array into class name - eg. array( 'My', 'Class', 'Name') to MyClassName
							$class_name = implode( '_', $class_to_filename );

							$class = __NAMESPACE__ . '\\' . $class_name;

							require_once $file;

							$status = false;

							$reflection_class = new \ReflectionClass( $class );

							$static_properties = $reflection_class->getStaticProperties();
							if ( key_exists( 'integration', $static_properties ) ) {
								$integration = $reflection_class->getStaticPropertyValue( 'integration' );
								$instance    = $reflection_class->newInstanceWithoutConstructor();
								$status      = $instance->plugin_active( 0, $integration );
							}

							if ( 1 === (int) $status ) {
								Utilities::add_class_instance( $class, new $class() );
							}
						}
					}
				}
			}
		}
	}

	/**
	 *
	 */
	public function boot_triggers_actions_closures() {

		global $uncanny_automator;

		// Check each directory
		foreach ( $this->auto_loaded_directories as $directory ) {
			// Get all files in directory
			if ( strpos( $directory, 'actions' ) || strpos( $directory, 'triggers' ) || strpos( $directory, 'tokens' ) || strpos( $directory, 'closures' ) ) {
				//$directory = str_replace( dirname( UAP_AUTOMATOR_FILE_ ) . '/', '', $directory_path );

				if ( file_exists( $directory ) ) {
					// Get all files in directory
					// remove parent directory, sub directory, and silence is golden index.php
					$files = array_diff( scandir( $directory ), array( '..', '.', 'index.php' ) );

					if ( $files ) {
						// Loop through all files in directory to create class names from file name
						foreach ( $files as $file ) {
							if ( strpos( $file, '.php' ) ) {
								// Remove file extension my-class-name.php to my-class-name
								$file_name = basename( $file, '.php' );

								// Split file name on - eg: my-class-name to array( 'my', 'class', 'name')
								$class_to_filename = explode( '-', $file_name );

								// Make the first letter of each word in array upper case - eg array( 'my', 'class', 'name') to array( 'My', 'Class', 'Name')
								$class_to_filename = array_map( function ( $word ) {
									return ucfirst( $word );
								}, $class_to_filename );

								// Implode array into class name - eg. array( 'My', 'Class', 'Name') to MyClassName
								$class_name = implode( '_', $class_to_filename );

								$class = __NAMESPACE__ . '\\' . strtoupper( $class_name );

								// We way want to include some class with the autoloader but not initialize them ex. interface class
								$skip_classes = apply_filters( 'Skip_class_initialization', array(), $directory, $files, $class, $class_name );
								if ( in_array( $class_name, $skip_classes ) ) {
									continue;
								}

								$reflection_class  = new \ReflectionClass( $class );
								$static_properties = $reflection_class->getStaticProperties();

								$status = 0;


								if ( key_exists( 'integration', $static_properties ) ) {
									$integration = $reflection_class->getStaticPropertyValue( 'integration' );
									$status      = $uncanny_automator->plugin_status->get( $integration );
								}

								if ( 1 === (int) $status && class_exists( $class ) ) {
									Utilities::add_class_instance( $class, new $class() );
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * @param $plugin_to_check
	 *
	 * @return bool
	 */
	public function required_plugin_active( $plugin_to_check ) {
		// Sanity check that there was a trigger passed
		if ( null === $plugin_to_check || ! is_string( $plugin_to_check ) ) {
			Utilities::log( 'ERROR: You are try to get a plugin\'s status without passing its proper integration code.', 'get_plugin_status ERROR', false, 'uap - errors' );

			return null;
		}
		$plugin_to_check = str_replace( [ '/actions', '/closures', '/triggers', '/tokens' ], '', $plugin_to_check );

		if ( is_dir( $plugin_to_check ) ) {
			// Its a directory name ... pull out the plugin name
			$raw    = explode( '/', $plugin_to_check );
			$plugin = strtolower( trim( basename( array_pop( $raw ) ) ) );
		} else {
			// if its not a directory name the we assume its the integration code
			$plugin = strtoupper( $plugin_to_check );
		}

		$active = 0;

		if ( ! empty( $plugin ) ) {
			$filter = "uncanny_automator_pro_maybe_add_{$plugin}_integration";
			$active = apply_filters( $filter, 0, $plugin );
		}

		return absint( $active );
	}


	/**
	 * @param $class
	 */
	public function require_triggers_actions( $class ) {

		// Remove Class's namespace eg: my_namespace/MyClassName to MyClassName
		$class = str_replace( __NAMESPACE__, '', $class );
		$class = str_replace( '\\', '', $class );

		// Replace _ with - eg. eg: My_Class_Name to My-Class-Name
		$class_to_filename = str_replace( '_', '-', $class );

		// Create file name that will be loaded from the classes directory eg: My-Class-Name to my-class-name.php
		$file_name = strtolower( $class_to_filename ) . '.php';
		/*echo '<pre>';
		print_r( $file_name );
		echo '</pre>';*/

		// Check each directory
		foreach ( $this->auto_loaded_directories as $directory ) {

			if ( strpos( $directory, 'actions' ) || strpos( $directory, 'triggers' ) || strpos( $directory, 'tokens' ) || strpos( $directory, 'closures' ) ) {
				//$directory = str_replace( dirname( UAP_AUTOMATOR_FILE_ ) . '/', '', $directory );
				if ( 'index.php' !== $file_name ) {
					$file_path = $directory . DIRECTORY_SEPARATOR . $file_name;


					// Does the file exist
					if ( file_exists( $file_path ) ) {

						// File found, require it
						require_once( $file_path );

						// You can cannot have duplicate files names. Once the first file is found, the loop ends.
						return;
					}
				}
			}
		}
	}
}
