<?php

namespace uncanny_learndash_groups;

// If this file is called directly, abort.

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class AdminMenu
 * @package uncanny_learndash_groups
 */
class GroupReportsInterface {

	/**
	 * The root path of the rest call
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $root_path = 'ulgm_management/v1/';

	/**
	 * Collection of variabls that can be dumped for debug
	 *
	 * @since 1.0.0
	 * @access private
	 * @var
	 */
	private $raw_data_log = array();


	/**
	 * An array of localized and filtered strings that are used in templates
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $ulgm_reporting_shortcode = array();

	/**
	 * An array of localized and filtered strings that are used in templates
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $transcript_page_url = '';

	/**
	 * Set the order of the course drop down
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $course_order = '';

	/**
	 * Set up the report for a specific user
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $user_id = 0;

	/**
	 * class constructor
	 */
	public function __construct() {

		// Enqueue Scripts for uo_group_management shortcode
		add_action( 'wp_enqueue_scripts', array( $this, 'uo_group_management_scripts' ) );

		/* ADD FILTERS ACTIONS FUNCTION */
		add_shortcode( 'uo_groups_course_report', array( $this, 'uo_group_course_report' ), 10, 2 );

		//register api class
		add_action( 'rest_api_init', array( $this, 'reporting_api' ) );

	}

	/**
	 * Loads all scripts and styles required by the shortcode
	 *
	 * @since 1.0.0
	 */
	function uo_group_management_scripts() {

		global $post;

		// Only add scripts if shortcode is present on page
		if ( Utilities::has_shortcode( $post, 'uo_groups_course_report' ) || Utilities::has_block( $post, 'uncanny-learndash-groups/uo-groups-course-report' ) ) {

			// Setup group management JS with localized WP Rest API variables @see rest-api-end-points.php
			wp_register_script( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );

			// API data
			$api_setup = array(
				'root'      => esc_url_raw( rest_url() . 'ulgm_management/v1/' ),
				'nonce'     => \wp_create_nonce( 'wp_rest' ),
				'localized' => $this->get_frontend_localized_strings()
			);

			wp_localize_script( 'ulgm-frontend', 'ulgmRestApiSetup', $api_setup );
			wp_enqueue_script( 'ulgm-frontend' );

			// Load styles
			wp_register_style( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.css' ), array(), Utilities::get_version() );
			$user_colors = Utilities::user_colors();
			wp_add_inline_style( 'ulgm-frontend', $user_colors );
			wp_enqueue_style( 'ulgm-frontend', $user_colors );

			// Load needed scripts
			wp_enqueue_script( 'ulgm-modernizr', Utilities::get_vendor( 'modernizr/js/modernizr.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jquery-lazyload', Utilities::get_vendor( 'jquery-lazyload/js/jquery.lazyload.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-core', Utilities::get_vendor( 'jplist/js/jplist.core.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-history-bundle', Utilities::get_vendor( 'jplist/js/jplist.history-bundle.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-pagination-bundle', Utilities::get_vendor( 'jplist/js/jplist.pagination-bundle.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-textbox-filter', Utilities::get_vendor( 'jplist/js/jplist.textbox-filter.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-sort-buttons', Utilities::get_vendor( 'jplist/js/jplist.sort-buttons.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-select2', Utilities::get_vendor( 'select2/js/select2.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_style( 'ulgm-font-awesome', Utilities::get_vendor( 'fontawesome/css/font-awesome.min.css' ), array(), Utilities::get_version() );
			wp_enqueue_style( 'ulgm-modal', Utilities::get_vendor( 'jquery-modal/css/jquery.modal.css' ), array(), Utilities::get_version() );
			wp_enqueue_style( 'ulgm-select2', Utilities::get_vendor( 'select2/css/select2.min.css' ), array(), Utilities::get_version() );


		}
	}

	/**
	 * Return the HTML template that is displayed by the shortcode
	 *
	 * @param array $attributes The attributes passed in the the shortcode
	 * @param string $content The content contained by the shortcode
	 *
	 * @return string $shortcode_template The HTML template loaded
	 * @since 1.0.0
	 *
	 */
	function uo_group_course_report( $attributes, $content = '' ) {

		$user_id = get_current_user_id();

		// Is the user logged in
		if ( ! $user_id ) {
			return __( 'Please log in to view the report.', 'uncanny-learndash-groups' );
		}

		// Is the user a group leader
		if ( ! user_can( $user_id, 'group_leader' ) ) {

			return __( 'You must be a group leader to view this report.', 'uncanny-learndash-groups' );

		}

		$request = shortcode_atts( array(
			'transcript-page-id' => 0,
			'course-order'       => ''
		), $attributes );

		if ( $request['transcript-page-id'] !== 0 ) {
			self::$transcript_page_url = apply_filters( 'group_reporting_transcript_url', get_permalink( absint( $request['transcript-page-id'] ) ), $request['transcript-page-id'] );
		}

		self::$course_order = $request['course-order'];

		if ( isset( $_GET['user_id'] ) ) {
			self::$user_id = absint( $_GET['user_id'] );
		}

		$this->localize_filter_globalize_text();

		ob_start();

		include( Utilities::get_template( 'frontend-uo_groups_course_report.php' ) );

		return ob_get_clean();

	}

	/*
	 * Register rest api endpoints
	 *
	 */
	function reporting_api() {

		register_rest_route( $this->root_path, '/get_group_courses/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'get_group_courses' )
		) );

		register_rest_route( $this->root_path, '/get_user_course_data/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'get_user_course_data' )
		) );

	}

	/*
	 * Get group of courses related to a LD Group
	 */
	public static function get_group_courses() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'get-courses' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}


		// Does the current user have permission
		$permission = apply_filters( 'group_management_add_user_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) && ! current_user_can( 'upload_plugins' ) ) {
			$data['message'] = __( 'You do not have permission to add users.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		$group_id = 0;

		// Was group id received
		if ( isset( $_POST['group-id'] ) ) {

			// is group a valid integer
			if ( ! absint( $_POST['group-id'] ) ) {
				$data['message'] = __( 'Group ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_leader_id = get_current_user_id();
			$user_group_ids  = learndash_get_administrators_group_ids( $group_leader_id, true );

			// is the current user able to administer this group
			if ( ! in_array( $_POST['group-id'], $user_group_ids ) ) {
				$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			$group_id = absint( $_POST['group-id'] );

		} else {
			$data['message'] = __( 'Group ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// send back group courses
		if ( 'get-courses' === $action ) {

			// Add the user and send out a welcome email
			$group_course_ids = learndash_group_enrolled_courses( $group_id );

			if ( empty( $group_course_ids ) ) {
				$data['message'] = __( 'This group does not have any courses', 'uncanny-learndash-groups' );
				wp_send_json_error( $data );
			}

			$posts_in = array_map( 'intval', $group_course_ids );

			$course_order = '';
			if ( isset( $_POST['course-order'] ) ) {
				$course_order = (string) $_POST['course-order'];
			}

			if ( empty( $course_order ) ) {
				$course_order = 'title';
			}

			if ( ! in_array( $course_order, [ 'ID', 'title', 'date', 'menu_order' ] ) ) {
				$course_order = 'title';
			}

			$args = array(
				'post_type'      => 'sfwd-courses',
				'post__in'       => $posts_in,
				'orderby'        => $course_order,
				'order'          => 'ASC',
				'posts_per_page' => - 1
			);

			$courses = new \WP_Query( $args );

			$data['reload']        = false;
			$data['call_function'] = 'populateCoursesDropDown';
			$data['function_vars'] = array(
				'group_courses' => $courses->posts,
				'group_id'      => $group_id
			);
			wp_send_json_success( $data );


		}

	}

	function get_user_course_data() {

		// Actions permitted by the pi call (colleced from input element with name action )
		$permitted_actions = array( 'get-user-data-courses' );

		// Was an action received, and is the actions allowed
		if ( isset( $_POST['action'] ) && in_array( $_POST['action'], $permitted_actions ) ) {

			$action = (string) $_POST['action'];

		} else {
			$action          = '';
			$data['message'] = __( 'Select an action.', 'uncanny-learndash-groups' );
			$data['testing'] = $_POST;
			wp_send_json_error( $data );
		}


		// Does the current user have permission
		$permission = apply_filters( 'group_management_add_user_permission', 'group_leader' );
		if ( ! current_user_can( $permission ) && ! current_user_can( 'upload_plugins' ) ) {
			$data['message'] = __( 'You do not have permission to add users.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		$group_id = 0;

		// Was group id received
		if ( isset( $_POST['course-group-id'] ) && isset( $_POST['user-id'] ) ) {

			// is group a valid integer
			if ( ! absint( $_POST['course-group-id'] ) && 'all' !== $_POST['course-group-id'] ) {
				$data['message'] = __( 'Course ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-course-id';
				wp_send_json_error( $data );
			}

			$group_leader_id  = get_current_user_id();
			$leader_group_ids = learndash_get_administrators_group_ids( $group_leader_id, true );
			$user_group_ids   = learndash_get_users_group_ids( absint( $_POST['user-id'] ), true );

			$user_group_ids = array_intersect( $leader_group_ids, $user_group_ids );

			// is the current user able to administer this group
			if ( ! in_array( $_POST['course-group-id'], $leader_group_ids ) && 'all' !== $_POST['course-group-id'] ) {
				$data['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

		} elseif ( ! isset( $_POST['course-group-id'] ) ) {
			$data['message'] = __( 'Group ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}
		
		if ( isset( $_POST['group-id'] ) && 'all' === $_POST['group-id'] ) {
			$group_id = (string) $_POST['course-group-id'];
		} else {
			$group_id = absint( $_POST['course-group-id'] );
		}
		
		$course_id = 0;
		// Was course id received
		if ( isset( $_POST['course-id'] ) ) {

			// is course a valid integer
			if ( ! absint( $_POST['course-id'] ) && 'all' !== $_POST['course-group-id'] ) {
				$data['message'] = __( 'Group ID must be a whole number.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-group-id';
				wp_send_json_error( $data );
			}

			if ( 'all' === $_POST['course-group-id'] ) {
				$group_course_ids = [];
				// get group courses
				foreach ( $user_group_ids as $_group_id ) {
					$group_course_ids = array_merge( $group_course_ids, learndash_group_enrolled_courses( $_group_id ) );
				}
				$group_course_ids = array_unique( $group_course_ids );

			} else {
				// get group courses
				$group_course_ids = learndash_group_enrolled_courses( $group_id );
			}


			// is the current user able to administer this group
			if ( ! in_array( $_POST['course-id'], $group_course_ids ) && 'all' !== $_POST['course-id'] ) {
				$data['message'] = __( 'This course is not part of this group.', 'uncanny-learndash-groups' );
				$data['error']   = 'invalid-course-id';
				wp_send_json_error( $data );
			}

			if ( 'all' === $_POST['course-id'] ) {
				$course_id = (string) $_POST['course-id'];
			} else {
				$course_id = absint( $_POST['course-id'] );
			}


		} else {
			$data['message'] = __( 'Course ID was not received. Reload page and try again.', 'uncanny-learndash-groups' );
			wp_send_json_error( $data );
		}

		// send back group courses
		if ( 'get-user-data-courses' === $action ) {

			$user_course_data = array();


			// Collect all user ids as array
			if ( 'all' === $group_id ) {
				// Get general user data$args = array(
				$args = [ 'include' => array( absint( $_POST['user-id'] ) ) ];

				$group_users                        = get_users( $args );
				$this->raw_data_log['$group_users'] = $group_users;

				$group_user_ids = [ absint( $_POST['user-id'] ) ];

				$user_course_ids = learndash_user_get_enrolled_courses( absint( $_POST['user-id'] ) );

				// We need to only query course that both the user and thr group leader have access too
				$group_course_ids = array_intersect( $group_course_ids, $user_course_ids );

			} else {
				// Get general user data
				$group_users                        = learndash_get_groups_users( $group_id );
				$this->raw_data_log['$group_users'] = $group_users;

				$group_user_ids = wp_list_pluck( $group_users, 'ID' );
			}


			if ( empty( $group_user_ids ) ) {
				$data['message'] = __( 'There are no users in this group', 'uncanny-learndash-groups' );
				$data['error']   = 'no-group-users';
				wp_send_json_error( $data );
			}
			// Get user course completions
			global $wpdb;
			$ld_activity_table      = $wpdb->prefix . 'learndash_user_activity';
			$ld_activity_meta_table = $wpdb->prefix . 'learndash_user_activity_meta';
			$group_user_ids_string  = implode( ',', $group_user_ids );

			// All variables are escaped and validated, prepare not needed
			$course_completions_query =
				"SELECT activity.user_id, activity.post_id, activity.activity_completed, meta.activity_meta_key, meta.activity_meta_value
				FROM $ld_activity_table activity
				JOIN $ld_activity_meta_table meta
				WHERE activity.activity_id = meta.activity_id
				AND activity.activity_type = 'course'
				AND activity.user_id IN ($group_user_ids_string)
				AND meta.activity_meta_key IN ('steps_total','steps_completed')";

			if ( 'all' !== $course_id ) {
				$course_completions_query .= " AND post_id = $course_id";
			}

			$course_completions = $wpdb->get_results( $course_completions_query );
			// set user ID as key for eay data selections
			$course_completions_by_id = array();
			foreach ( $course_completions as $key => $course_completion ) {
				if ( 'all' !== $course_id ) {
					$new_key = (int) $course_completion->user_id;
				} else {
					$new_key                                         = (int) $course_completion->post_id;
					$course_completions_by_id[ $new_key ]['user_id'] = (int) $course_completion->user_id;
				}

				$meta_key   = $course_completion->activity_meta_key;
				$meta_value = $course_completion->activity_meta_value;

				//if ( isset( $course_completions_by_id[ $new_key ] ) ) {
				$activity_completed = $course_completion->activity_completed;
				if ( ! $activity_completed ) {
					$activity_completed = '';
				}
				$course_completions_by_id[ $new_key ]['activity_completed'] = $activity_completed;
				//}

				$course_completions_by_id[ $new_key ][ $meta_key ] = $meta_value;

			}
			$this->raw_data_log['$course_completions'] = $course_completions;

			// Collect Quiz Averages
			//$course_quiz_average                        = $this->get_average_quiz_result( $course_id, $group_user_ids_string );
			//$this->raw_data_log['$course_quiz_average'] = $course_quiz_average;


			if ( 'all' === $course_id ) {

				$args = array(
					'numberposts' => 999,
					'include'     => array_values( array_map( 'intval', $group_course_ids ) ),
					'post_type'   => 'sfwd-courses'
				);

				global $wpdb;
				$courses      = get_posts( $args );
				$testing      = $wpdb->last_query;
				$course_by_ID = [];
				foreach ( $courses as $course ) {
					$course_by_ID[ $course->ID ] = $course;
				}

				// Join all data
				foreach ( $group_course_ids as $group_course_id ) {

					$user_id    = $group_users[0]->ID;
					$first_name = get_user_meta( $user_id, 'first_name', true );
					$last_name  = get_user_meta( $user_id, 'last_name', true );
					$user_email = $group_users[0]->data->user_email;
					//$quiz_average      = 0;
					$date_completed    = '';
					$percent_completed = '0%';

					$group_course_id = (string) $group_course_id;

//					if ( isset( $course_quiz_average[ $group_course_id ] ) && isset( $course_quiz_average[ $group_course_id ]['quiz_average'] ) ) {
//						$quiz_average = $course_quiz_average[ $group_course_id ]['quiz_average'];
//					}

					if ( isset( $course_completions_by_id[ $group_course_id ] ) && isset( $course_completions_by_id[ $group_course_id ]['activity_completed'] ) ) {

						// In raw situation a course has all lessons completed but completion is never triggered
						if ( empty( $course_completions_by_id[ $group_course_id ]['activity_completed'] ) ) {
							$date_completed = '';
						} else {
							$date_completed = learndash_adjust_date_time_display( $course_completions_by_id[ $group_course_id ]['activity_completed'] );
						}

					}
					//$date_completed = json_encode( $course_completions_by_id );
					if ( isset( $course_completions_by_id[ $group_course_id ] )
					     && isset( $course_completions_by_id[ $group_course_id ]['steps_completed'] )
					     && isset( $course_completions_by_id[ $group_course_id ]['steps_total'] )
					) {
						$percent_completed = ceil( $course_completions_by_id[ $group_course_id ]['steps_completed'] / $course_completions_by_id[ $group_course_id ]['steps_total'] * 100 ) . '%';
					}

					$user_course_data[] = array(
						'user_id'           => $user_id,
						//'display_name' => $display_name,
						'first_name'        => $first_name,
						'last_name'         => $last_name,
						'user_email'        => $user_email,
						//'quiz_average'      => $quiz_average,
						'date_completed'    => $date_completed,
						'percent_completed' => $percent_completed,
						'course_name'       => $course_by_ID[ (int) $group_course_id ]->post_title
					);


				}
			} else {
				// Join all data
				foreach ( $group_users as $user ) {

					$user_id = $user->ID;
					//$display_name      = $user->data->user_nicename;
					$first_name = get_user_meta( $user_id, 'first_name', true );
					$last_name  = get_user_meta( $user_id, 'last_name', true );
					$user_email = $user->data->user_email;
					//$quiz_average      = 0;
					$date_completed    = '';
					$percent_completed = '0%';

//					if ( isset( $course_quiz_average[ $user_id ] ) && isset( $course_quiz_average[ $user_id ]['quiz_average'] ) ) {
//						$quiz_average = $course_quiz_average[ $user_id ]['quiz_average'];
//					}

					if ( isset( $course_completions_by_id[ $user_id ] ) && isset( $course_completions_by_id[ $user_id ]['activity_completed'] ) ) {

						// In raw situation a course has all lessons completed but completion is never triggered
						if ( empty( $course_completions_by_id[ $user_id ]['activity_completed'] ) ) {
							$date_completed = '';
						} else {

							$date_completed = learndash_adjust_date_time_display( $course_completions_by_id[ $user_id ]['activity_completed'] );

						}
					}

					if ( isset( $course_completions_by_id[ $user_id ] )
					     && isset( $course_completions_by_id[ $user_id ]['steps_completed'] )
					     && isset( $course_completions_by_id[ $user_id ]['steps_total'] )
					) {
						$percent_completed = ceil( $course_completions_by_id[ $user_id ]['steps_completed'] / $course_completions_by_id[ $user_id ]['steps_total'] * 100 ) . '%';
					}

					$user_course_data[] = array(
						'user_id'           => $user_id,
						//'display_name' => $display_name,
						'first_name'        => $first_name,
						'last_name'         => $last_name,
						'user_email'        => $user_email,
						//'quiz_average'      => $quiz_average,
						'date_completed'    => $date_completed,
						'percent_completed' => $percent_completed
					);


				}
			}


			$this->raw_data_log['$user_course_data'] = $user_course_data;


			$data['reload']        = false;
			$data['call_function'] = 'populateReportTable';
			$data['function_vars'] = array(
				'user_course_data'          => $user_course_data,
				'$course_completions_by_id' => $course_completions_by_id,
				'$course_completions_query' => $course_completions_query,
				'$course_completions'       => $course_completions,
				'$group_course_ids'         => $group_course_ids,
				'$group_users'              => $group_users,
				//'$course_by_ID' => $course_by_ID,
				//'$args' => $args,
				//'$courses' => $courses,
				//'$testing' => $testing
			);

			//Utilities::log( $this->raw_data_log, 'Group Report - User Course Data');
			wp_send_json_success( $data );
		}

	}

	/**
	 * Collect the highest quiz results and averages
	 *
	 * @param int $course_id
	 *
	 * @return array $user_quiz_data
	 */
	private function get_average_quiz_result( $course_id, $group_user_ids_string ) {

		global $wpdb;

		// Get all course quiz
		$args = [
			'post_type'   => 'sfwd-quiz',
			'post_status' => 'publish',
			'meta_key'    => 'course_id',
			'meta_value'  => $course_id
		];

		$course_quiz_list = new \WP_Query( $args );

		// Collect all user ids as array
		$course_quiz_list_ids                        = wp_list_pluck( $course_quiz_list->posts, 'ID' );
		$this->raw_data_log['$course_quiz_list_ids'] = $course_quiz_list_ids;

		// Get all group's users quiz data
		// All variables are escaped and validated, prepare not needed
		$quiz_data_query                        = "SELECT user_id, meta_value FROM $wpdb->usermeta WHERE user_id IN ($group_user_ids_string) AND meta_key = '_sfwd-quizzes'";
		$quiz_data                              = $wpdb->get_results( $quiz_data_query );
		$this->raw_data_log['$quiz_data_query'] = $quiz_data_query;
		$this->raw_data_log['$quiz_data']       = $quiz_data;


		// Store collected quiz percentages and averages
		$user_quiz_data = array();

		// Loop through all user's quiz results
		foreach ( $quiz_data as $user_data ) {

			// Set User ID
			$user_id = $user_data->user_id;

			// Create array of all quiz attempts for the user
			$all_quiz_data = maybe_unserialize( $user_data->meta_value );

			// Loop through all user's quizzes
			foreach ( $all_quiz_data as $quiz_data ) {

				// Validate that the quiz is in the current course
				if ( in_array( (int) $quiz_data['quiz'], $course_quiz_list_ids ) ) {

					// Validate that if there is quiz percentage set for the quiz based on the user
					if ( isset( $user_quiz_data[ $user_id ]['quiz_percentage'][ (int) $quiz_data['quiz'] ] ) ) {

						// There is a percentage, we are calculating highest scores for each quiz, reset the quiz average if this quiz attempt is higher
						if ( $user_quiz_data[ $user_id ]['quiz_percentage'][ (int) $quiz_data['quiz'] ] < $quiz_data['percentage'] ) {
							$user_quiz_data[ $user_id ]['quiz_percentage'][ (int) $quiz_data['quiz'] ] = $quiz_data['percentage'];
						}

					} else {
						// store the quiz percentage
						$user_quiz_data[ $user_id ]['quiz_percentage'][ (int) $quiz_data['quiz'] ] = $quiz_data['percentage'];
					}

				}

			}

		}

		// Calculate the quiz average for each user
		foreach ( $user_quiz_data as $user_id => $data ) {
			$scores                                     = $data['quiz_percentage'];
			$user_quiz_data[ $user_id ]['quiz_average'] = ceil( array_sum( $scores ) / count( $scores ) );
		}

		return $user_quiz_data;

	}

	/**
	 * Filter and localize all text, then set as global for use in template file
	 *
	 * @since 1.0.0
	 */
	function localize_filter_globalize_text() {

		// Enrolled user table headers
		self::$ulgm_reporting_shortcode['table']['reporting']['headers'][0] = array(
			'title' => __( 'First Name', 'uncanny-learndash-groups' ),
			'slug'  => 'first-name'
		);

		// Enrolled user table headers
		self::$ulgm_reporting_shortcode['table']['reporting']['headers'][1] = array(
			'title' => __( 'Last Name', 'uncanny-learndash-groups' ),
			'slug'  => 'last-name'
		);

		self::$ulgm_reporting_shortcode['table']['reporting']['headers'][2] = array(
			'title' => __( 'Email', 'uncanny-learndash-groups' ),
			'slug'  => 'email'
		);

		self::$ulgm_reporting_shortcode['table']['reporting']['headers'][3] = array(
			'title' => __( '% Complete', 'uncanny-learndash-groups' ),
			'slug'  => 'percent-complete'
		);

		self::$ulgm_reporting_shortcode['table']['reporting']['headers'][4] = array(
			'title' => __( 'Date Completed', 'uncanny-learndash-groups' ),
			'slug'  => 'date-completed'
		);

		if ( absint( self::$user_id ) ) {
			self::$ulgm_reporting_shortcode['table']['reporting']['headers'][5] = array(
				'title' => __( 'Course Name', 'uncanny-learndash-groups' ),
				'slug'  => 'course-name'
			);
		}


		self::$ulgm_reporting_shortcode['text']['group_management_link'] = SharedFunctions::get_group_management_page_id( true );

		self::$ulgm_reporting_shortcode['text']['group_management'] = __( 'Back to Group Management', 'uncanny-learndash-groups' );

		if ( '' !== self::$transcript_page_url ) {
			self::$ulgm_reporting_shortcode['table']['reporting']['headers'][4] = array(
				'title' => __( 'Transcript', 'uncanny-learndash-groups' ),
				'slug'  => 'transcript'
			);
		}


	}

	private function get_frontend_localized_strings() {

		$localized_strings = array();

		$localized_strings['selectCourse'] = sprintf( __( 'Select %s', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'course' ) );

		$localized_strings['noCourse'] = __( 'No courses Available', 'uncanny-learndash-groups' );

		$localized_strings = apply_filters( 'group-report-table-strings', $localized_strings );


		return $localized_strings;
	}


}