<?php

namespace uncanny_learndash_groups;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class Sample
 * @package uncanny_pro_toolkit
 */
class GroupQuizReport {

	/**
	 * Rest API root path
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $root_path = 'ulgm_quiz_report/v1/';

	static $ulgm_reporting_shortcode = array();

	/**
	 * Rest API root path
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	static $group_drop_downs = false;

	/**
	 * Course order by
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $course_orderby = '';

	/**
	 * Course order
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $course_order = '';

	/**
	 * Course order by
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $quiz_orderby = '';

	/**
	 * Course order
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $quiz_order = '';


	/**
	 * Class constructor
	 */
	public function __construct() {

		add_action( 'plugins_loaded', array( $this, 'run_frontend_hooks' ) );

		//register api class
		add_action( 'rest_api_init', array( $this, 'uo_api' ) );
	}

	/*
	 * Initialize frontend actions and filters
	 */
	public function run_frontend_hooks() {

		//add_action( 'wp_enqueue_scripts', array( $this, 'report_scripts' ) );
		add_shortcode( 'uo_groups_quiz_report', array( $this, 'display_quiz_report' ) );
	}

	/*
	 * Display the shortcode
	 * @param array $attributes
	 *
	 * @return string $html header and table
	 */
	public function display_quiz_report( $request ) {

		$request = shortcode_atts( array(
			'course-orderby' => 'title',
			'course-order'   => 'ASC',
			'quiz-orderby'   => 'title',
			'quiz-order'     => 'ASC',
		), $request );

		$course_orderby = $request['course-orderby'];

		if ( ! in_array( $course_orderby, [ 'ID', 'title', 'date', 'menu_order' ] ) ) {
			$this->course_orderby = 'title';
		} else {
			$this->course_orderby = $request['course-orderby'];
		}

		$course_order = $request['course-order'];

		if ( ! in_array( $course_order, [ 'ASC', 'DESC' ] ) ) {
			$this->course_order = 'ASC';
		} else {
			$this->course_order = $request['course-order'];
		}

		$quiz_orderby = $request['quiz-orderby'];

		if ( ! in_array( $quiz_orderby, [ 'ID', 'title', 'date', 'menu_order' ] ) ) {
			$this->quiz_orderby = 'title';
		} else {
			$this->quiz_orderby = $request['quiz-orderby'];
		}

		$quiz_order = $request['quiz-order'];

		if ( ! in_array( $quiz_order, [ 'ASC', 'DESC' ] ) ) {
			$this->quiz_order = 'ASC';
		} else {
			$this->quiz_order = $request['quiz-order'];
		}


		$html = $this->generate_quiz_report_html();

		$this->report_scripts();

		return $html;
	}

	/*
	 * Generate transcript HTML Output
	 *
	 * @return string
	 */
	function generate_quiz_report_html() {

		//  Build Table Data
		$table             = [];
		$table['headings'] = [
			'Quiz',
			'Quiz Sore',
			'Answers',
		];

		self::$ulgm_reporting_shortcode['text']['group_management_link'] = SharedFunctions::get_group_management_page_id( true );

		self::$ulgm_reporting_shortcode['text']['group_management'] = __( 'Back to Group Management', 'uncanny-learndash-groups' );

		return $this->create_quiz_table( $table['headings'] );
	}

	function create_quiz_table( $headings ) {
		global $post;

		$user_id = get_current_user_id();

		// Is the user logged in
		if ( ! $user_id ) {
			return __( 'Please log in to view the report.', 'uncanny-learndash-groups' );
		}

		// Is the user a group leader
		if ( user_can( $user_id, 'group_leader' ) ) {

			// Load Selection options for group and quiz list
			$group_drop_downs = $this->get_groups_drop_downs( $user_id );

		} else {
			return __( 'You must be a group leader to view this report.', 'uncanny-learndash-groups' );
		}

		// Create HTML table headings
		$headings_html = '';
		foreach ( $headings as $heading ) {
			$headings_html .= '<th>' . $heading . '</th>';
		}

		ob_start();

		if ( defined( 'LEARNDASH_LMS_PLUGIN_URL' ) && function_exists( 'learndash_is_active_theme' ) && learndash_is_active_theme( 'ld30' ) ) {
			$icon = LEARNDASH_LMS_PLUGIN_URL . 'themes/legacy/templates/images/statistics-icon-small.png';
			?>
			<style>
				.statistic_icon {
					background: url(<?php echo $icon; ?>) no-repeat scroll 0 0 transparent;
					width: 23px;
					height: 23px;
					margin: auto;
					background-size: 23px;
				}
			</style>
			<?php
		}

		// Load Learndash Quiz modal window assets and localized data
		$this->get_ld_modal();

		\LD_QuizPro::showModalWindow();


		?>

		<div class="uo-groups uo-quiz-report">
			<?php if ( ! empty( SharedFunctions::get_group_management_page_id() ) && $post->ID === SharedFunctions::get_group_quiz_report_page_id() ): ?>
				<div class="uo-row uo-groups-section uo-groups-report-go-back">
					<div class="uo-groups-actions">
						<div class="group-management-buttons">
							<button class="ulgm-link uo-btn uo-left uo-btn-arrow-left"
									onclick="location.href='<?php echo GroupQuizReport::$ulgm_reporting_shortcode['text']['group_management_link']; ?>'"
									type="button">
								<?php echo GroupQuizReport::$ulgm_reporting_shortcode['text']['group_management']; ?>
							</button>
						</div>
					</div>
				</div>
			<?php endif; ?>
			<div class="uo-row uo-groups-section uo-groups-selection">

				<div class="group-management-form">

					<div class="uo-groups-select-filters">
						<div class="uo-row uo-groups-select-filter">
							<div class="uo-select">
								<label><?php _e( 'Group', 'uncanny-learndash-groups' ); ?></label>
								<select class="change-group-management-form"
										id="uo-group-report-group"><?php echo $group_drop_downs['groups']; ?></select>
							</div>
						</div>

						<div class="uo-row uo-groups-select-filter">
							<div class="uo-select">
								<label><?php _e( \LearnDash_Custom_Label::get_label( 'course' ), 'uncanny-learndash-groups' ); ?></label>
								<select class="change-group-management-form" style="display:none;"
										id="uo-group-report-courses"><?php echo $group_drop_downs['courses']; ?></select>
								<div id="uo-group-report-nocourses" class="group-management-rest-message"
									 style="display: none;"><?php echo __( 'No courses found.', 'uncanny-learndash-groups' ); ?></div>
							</div>
						</div>

						<div class="uo-row uo-groups-select-filter">
							<div class="uo-select">
								<label><?php _e( \LearnDash_Custom_Label::get_label( 'quiz' ), 'uncanny-learndash-groups' ); ?></label>
								<select class="change-group-management-form" style=""
										id="uo-group-report-quizzes"><?php echo $group_drop_downs['quizzes']; ?></select>
								<div id="uo-group-report-noquizzes" class="group-management-rest-message"
									 style="display: none;"><?php echo __( 'No quizzes found.', 'uncanny-learndash-groups' ); ?></div>
							</div>
						</div>
					</div>

				</div>
			</div>

			<div class="uo-row uo-groups-table">
				<table id="uo-quiz-report-table-hidden" class="display responsive no-wrap uo-table-datatable"
					   cellspacing="0" width="100%"></table>

				<table id="uo-quiz-report-table" class="display responsive no-wrap uo-table-datatable" cellspacing="0"
					   width="100%"></table>
			</div>
		</div>

		<style>
			.uo-groups .uo-select select.h3-select {
				background: none !important;
				border: none;
				-webkit-box-shadow: none;
				box-shadow: none;
				font-size: 18px;
				font-weight: bold;
				padding-left: 0;
				padding-top: 0;
				/*for firefox*/
				-moz-appearance: none;
				/*for chrome*/
				-webkit-appearance: none;
			}

			/*for IE10*/
			.uo-groups .uo-select select.h3-select::-ms-expand {
				display: none;
			}

			.showme {
				display: block !important;
			}
		</style>

		<script>
            jQuery(document).ready(function () {

                let groupSelect = jQuery('#uo-group-report-group');
                let groupSelectOptions = jQuery('#uo-group-report-group option');

                if (2 === groupSelectOptions.length) {
                    groupSelect.addClass('h3-select');
                    let groupId = jQuery(groupSelect.find('option')[1]).val();
                    window.ulgmGroupManagement.reporting.groupId = groupId;
                    groupSelect.val(groupId).trigger('change');

                    groupSelect.prop('disabled', 'disabled');

                    let courseGroupQuizzes = groupQuizReportSetup.courseGroupQuizzes[groupId];
                    let coursesSelect = jQuery('#uo-group-report-courses');
                    let coursesSelectOptions = jQuery('#uo-group-report-courses option');

                    if (2 === coursesSelectOptions.length) {

                        coursesSelect.show();
                        coursesSelect.addClass('h3-select');
                        // below line commented BY_AC
                        //let courseId = jQuery( coursesSelect.find('option')[1] ).val();
                        let courseId = jQuery(coursesSelect.find('option')[1]).val();
                        window.ulgmGroupManagement.reporting.courseId = courseId;
                        coursesSelect.val(courseId).trigger('change');

                        coursesSelect.prop('disabled', 'disabled');

                        jQuery('#uo-group-report-quizzes').show();
                        jQuery('#uo-group-report-quizzes option').hide();
                        jQuery('#uo-group-report-quizzes option').removeAttr('selected');
                        // below line commented BY_AC
                        //jQuery('#uo-group-report-quizzes option[value=0]').attr("selected", "selected");
                        jQuery('#uo-group-report-quizzes option[value=0]').show();

                        var groupsQuizzes = groupQuizReportSetup.relationships[window.ulgmGroupManagement.reporting.groupId][courseId];

                        if (typeof groupsQuizzes !== 'undefined' && groupsQuizzes.length > 0) {
                            jQuery.each(groupsQuizzes, function (key, quizId) {
                                jQuery('#uo-group-report-quizzes option[value=' + quizId + ']').addClass('showme');
                                jQuery('#uo-group-report-quizzes option[value=' + quizId + ']').css('display', 'block');
                                jQuery('#uo-group-report-quizzes option[value=' + quizId + ']').show();
                            });
                            // need a delay and let jquery finish actions in each loop
                            setTimeout(function () {
                                jQuery('#uo-group-report-quizzes').trigger('change');
                            }, 200);
                        } else {
                            jQuery('#uo-group-report-quizzes option').hide();
                            jQuery('#uo-group-report-quizzes option[value=' + 0 + ']').show();
                            //jQuery('#uo-group-report-noquizzes').show();
                        }
                    } else {
                        coursesSelect.show();
                        jQuery('#uo-group-report-courses option').hide();
                        jQuery('#uo-group-report-courses option').removeAttr('selected');
                        jQuery('#uo-group-report-courses option[0]').attr("selected", "selected");
                        jQuery('#uo-group-report-courses option[0]').show();
                        jQuery('#uo-group-report-courses option[value=' + 0 + ']').show();
                        jQuery.each(courseGroupQuizzes, function (key, courseId) {
                            jQuery('#uo-group-report-courses option[value=' + courseId + ']').show();
                        });
                        setTimeout(function () {
                            jQuery('#uo-group-report-courses').trigger('change');
                        }, 400);
                    }
                    window.ulgmGroupManagement.reporting.groupId = groupId;
                } else {
                    // need a delay and let jquery finish actions in each loop
                    setTimeout(function () {
                        let groupId = jQuery(groupSelect.find('option')[0]).val();
                        window.ulgmGroupManagement.reporting.groupId = groupId;
                        groupSelect.val(groupId).trigger('change');

                    }, 200);
                }
            })
		</script>

		<?php

		return ob_get_clean();
	}

	/**
	 * Loads Javascript files and variables used for the Quiz statistics modal
	 * We are leveraging it for our reports
	 */
	function get_ld_modal() {

		global $learndash_assets_loaded;

		if ( isset( $learndash_assets_loaded['scripts']['learndash_template_script_js'] ) ) {

			wp_dequeue_script( 'learndash_template_script_js' );

			wp_enqueue_script( 'learndash_template_script_js_2', Utilities::get_asset( 'frontend', 'learndash_template_script.js' ), array( 'jquery' ), LEARNDASH_VERSION, true );

			wp_enqueue_script( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.js' ), [ 'jquery' ], Utilities::get_version() );

			$data            = array();
			$data['ajaxurl'] = admin_url( 'admin-ajax.php' );
			$data            = array( 'json' => json_encode( $data ) );

			wp_localize_script( 'ulgm-frontend', 'sfwd_data', $data );
		}
	}

	/*
	 * Get all Groups the group leader is an administrator of
	 * @since
	 *
	 * @param int $group_leader_id
	 *
	 * @return string html
	 */
	function get_groups_drop_downs( $user_id = 0 ) {

		if ( false !== self::$group_drop_downs ) {
			return self::$group_drop_downs;
		}

		$user_id = get_current_user_id();

		if ( ! user_can( $user_id, 'group_leader' ) ) {
			return false;
		}

		// User is a group leader, get users groups ... We already verified that the user is already a group leader
		$user_groups = learndash_get_administrators_group_ids( $user_id, true );

		if ( empty( $user_groups ) ) {
			return __( 'You are not a leader of any groups.', 'uncanny-learndash-groups' );
		}

		// LD returns a array of IDs as strings, refactor to Int
		$posts_in = array_map( 'intval', $user_groups );

		$args = array(
			'post_type'      => 'groups',
			'post__in'       => $posts_in,
			'posts_per_page' => - 1,
			'orderby'        => 'title',
			'order'          => 'ASC'
		);


		$args = apply_filters( 'quiz_group_dropdown', $args, $user_id, $posts_in );

		$group_post_objects = new \WP_Query( $args );

		$drop_down['groups'] = '<option value="0">' . __( 'Select Group', 'uncanny-learndash-groups' ) . '</option>';
		//$drop_down['groups'] = '';
		$drop_down['quizzes_objects'] = array();

		// Collect all the quiz IDs so we can query then altogether
		$course_ids = array();
		$quiz_ids   = array();

		if ( $group_post_objects->have_posts() ) {
			while ( $group_post_objects->have_posts() ) {
				$group_post_objects->the_post();
				$drop_down['groups'] .= '<option value="' . get_the_ID() . '">' . get_the_title() . '</option>';

				$group_quizzes                                = $this->group_quizzes( get_the_ID() );
				$drop_down['quizzes_objects'][ get_the_ID() ] = $group_quizzes['group_quiz_ids'];

				$drop_down['course_quizzes_objects'][ get_the_ID() ] = $group_quizzes['group_course_quizzes'];

				$course_ids = array_merge( $course_ids, $group_quizzes['group_course_quizzes'] );
				$quiz_ids   = array_merge( $quiz_ids, $group_quizzes['group_quiz_ids'] );

				$drop_down['relationships'][ get_the_ID() ] = $group_quizzes['relationships'];


			}
			/* Restore original Post Data */
			wp_reset_postdata();
		} else {
			// no posts found
			$drop_down['groups'] = '<option value="0">' . __( 'No Groups', 'uncanny-learndash-groups' ) . '</option>';
		}

		// Get Courses
		$course_ids = array_unique( $course_ids );

		$courses = $this->get_objects( $course_ids, 'sfwd-courses', $this->course_orderby, $this->course_order );

		if ( ! empty( $courses ) ) {
			// below line commented BY_AC
			$drop_down['courses'] = '<option value="0">' . sprintf( __( 'Select %s', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'course' ) ) . '</option>';
			//$drop_down['courses'] = '';
			foreach ( $courses as $course ) {
				$drop_down['courses'] .= '<option value="' . $course->ID . '"  style="display:none">' . $course->post_title . '</option>';;
			}
		} else {
			$drop_down['courses'] = '<option value="0">' . __( 'No Courses', 'uncanny-learndash-groups' ) . '</option>';
		}

		// Get Quizzes
		$quiz_ids = array_unique( $quiz_ids );

		$quizzes = $this->get_objects( $quiz_ids, 'sfwd-quiz', $this->quiz_orderby, $this->quiz_order );

		if ( ! empty( $quizzes ) ) {
			// below line commented BY_AC
			$drop_down['quizzes'] = '<option value="0">' . sprintf( __( 'Select %s', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'quiz' ) ) . '</option>';
			//$drop_down['quizzes'] = '';
			foreach ( $quizzes as $quiz ) {
				$drop_down['quizzes'] .= '<option value="' . $quiz->ID . '" style="display:none">' . $quiz->post_title . '</option>';;
			}
		} else {
			$drop_down['quizzes'] = '<option value="0">' . __( 'No Quizzes', 'uncanny-learndash-groups' ) . '</option>';
		}


		// Cache results so we don't re-query
		self::$group_drop_downs = $drop_down;

		return $drop_down;
	}

	/**
	 * Get groups course quizzes
	 *
	 * @param int $group_id
	 *
	 * @return mixed
	 */
	function group_quizzes( $group_id = 0 ) {
		$group_quiz_ids = array();

		$relationships = array();
		if ( ! empty( $group_id ) ) {

			$group_course_ids = learndash_group_enrolled_courses( intval( $group_id ) );

			if ( ! empty( $group_course_ids ) ) {
				foreach ( $group_course_ids as $course_id ) {
					$group_quiz_query_args = apply_filters( 'ulgm_group_quiz_query_args', array(
						'post_type'  => 'sfwd-quiz',
						'nopaging'   => true,
						'orderby'    => 'title',
						'order'      => 'ASC',
						'fields'     => 'ids',
						'meta_query' => array(
							array(
								'relation' => 'OR',
								array(
									'key'     => 'course_id',
									'value'   => $course_id,
									'compare' => '=',
								),
								array(
									'key'     => 'ld_course_' . $course_id,
									'value'   => $course_id,
									'compare' => '=',
								)
							)
						)
					) );


					$group_quiz_query = new \WP_Query( $group_quiz_query_args );

					if ( ! empty( $group_quiz_query->posts ) ) {
						$group_quiz_ids = array_merge( $group_quiz_ids, $group_quiz_query->posts );
						$group_quiz_ids = array_unique( $group_quiz_ids );
						if ( isset( $relationships[ $course_id ] ) ) {
							$relationships[ $course_id ] = array_unique( array_merge( $relationships[ $course_id ], $group_quiz_query->posts ) );

						} else {
							$relationships[ $course_id ] = $group_quiz_query->posts;

						}

					}
				}
			}
		}

		return array(
			'group_quiz_ids'       => $group_quiz_ids,
			'group_course_quizzes' => $group_course_ids,
			'relationships'        => $relationships
		);
	}

	/**
	 * Get all quiz post objects
	 *
	 * @param array $quiz_ids
	 *
	 * @return array $_quizzes
	 */
	function get_objects( $ids, $post_type, $order_by = 'title', $order = 'ASC' ) {

		if ( empty( $order_by ) ) {
			$order_by = 'title';
		}

		if ( empty( $order ) ) {
			$order = 'ASC';
		}

		if ( empty( $ids ) ) {
			return [];
		}

		$args = array(
			'post_type'      => $post_type,
			'post__in'       => $ids,
			'posts_per_page' => - 1,
			'orderby'        => $order_by,
			'order'          => $order
		);

		$quizzes = get_posts( $args );

		// Set the Key as the post ID so we don't have to run a nested loop
		$_quizzes = array();
		foreach ( $quizzes as $quiz ) {
			$_quizzes[ $quiz->ID ] = $quiz;
		}

		return $_quizzes;

	}


	/*
	 * Register rest api endpoints
	 *
	 */
	public function uo_api() {

		register_rest_route( $this->root_path, '/get_quiz_data/', array(
			'methods'  => 'POST',
			'callback' => array( $this, 'get_quiz_data' )
		) );

	}

	function get_quiz_data() {

		if ( false === Utilities::get_debug_mode() ) {
			error_reporting( 0 );
		}

		$data = $_POST;

		// validate inputs
		$quiz_ID = absint( $data['quizId'] );


		// if any of the values are 0 then they didn't validate
		if ( 0 === $quiz_ID ) {
			$return_object['message'] = __( 'invalid course id supplied', 'uncanny-learndash-groups' );;
			$return_object['groupId'] = $data['courseId'];

			return $return_object;
		}

		// validate inputs
		$group_ID = absint( $data['groupId'] );

		// if any of the values are 0 then they didn't validate
		if ( 0 === $group_ID ) {
			$return_object['message'] = __( 'invalid group id supplied', 'uncanny-learndash-groups' );
			$return_object['groupId'] = $data['groupId'];

			return $return_object;
		}

		$quiz_table = $this->quiz_table( $quiz_ID, $group_ID );

		return $quiz_table;

	}

	/**
	 * Return html for the quizzes table
	 *
	 * @param $quiz_ID
	 * @param $group_ID
	 *
	 * @return array
	 *
	 */
	function quiz_table( $quiz_ID, $group_ID ) {

		global $learndash_shortcode_used;

		$group_users = learndash_get_groups_user_ids( $group_ID );

		$user_data = Utilities::get_users_with_meta( array( '_sfwd-quizzes', 'first_name', 'last_name' ) );

		$data      = array();
		$html_vars = array();

		// $user_data returned all users data. Let remove all non-members of group
		// ToDo get_users_with_meta() can be modified to only get group users for a performance tweak
		foreach ( $user_data['results'] as $user ) {
			if ( in_array( (int) $user['ID'], $group_users ) ) {
				$data[ $user['ID'] ] = $user;
			}
		}

		$learndash_shortcode_used = true;

		foreach ( $data as $user_id => $user ) {

			$quiz_attempts_meta = empty( $user['_sfwd-quizzes'] ) ? false : $user['_sfwd-quizzes'];

			if ( ! empty( $quiz_attempts_meta ) ) {


				$quiz_attempts_meta = maybe_unserialize( $quiz_attempts_meta );

				//Utilities::log( $quiz_attempts_meta, '$quiz_attempts_meta', true, 'quiz-report' );

				foreach ( $quiz_attempts_meta as $quiz_attempt ) {

					if ( (int) $quiz_attempt['quiz'] !== $quiz_ID ) {
						continue;
					}

					$modal_link = '';

					$quiz_attempt['percentage'] = ! empty( $quiz_attempt['percentage'] ) ? $quiz_attempt['percentage'] : ( ! empty( $quiz_attempt['count'] ) ? $quiz_attempt['score'] * 100 / $quiz_attempt['count'] : 0 );

					if ( ( isset( $quiz_attempt['has_graded'] ) ) && ( true === $quiz_attempt['has_graded'] ) && ( true === \LD_QuizPro::quiz_attempt_has_ungraded_question( $quiz_attempt ) ) ) {
						$score = _x( 'Pending', 'Pending Certificate Status Label', 'learndash' );
					} else {
						$score = round( $quiz_attempt['percentage'], 2 ) . __( '%', 'uncanny-learndash-groups' );
					}


					if ( intval( $quiz_attempt['statistic_ref_id'] ) ) {
						$modal_link = '<a class="user_statistic"
									     data-statistic_nonce="' . wp_create_nonce( 'statistic_nonce_' . $quiz_attempt['statistic_ref_id'] . '_' . get_current_user_id() . '_' . $user['ID'] ) . '"
									     data-user_id="' . $user['ID'] . '"
									     data-quiz_id="' . $quiz_attempt['pro_quizid'] . '"
									     data-ref_id="' . intval( $quiz_attempt['statistic_ref_id'] ) . '"
									     href="#">';
						$modal_link .= '<div class="statistic_icon"></div>';
						$modal_link .= '</a>';
					} else {
						$modal_link = __( 'No Stats Recorded', 'uncanny-learndash-groups' );
					}


					$date = learndash_adjust_date_time_display( $quiz_attempt['time'] );

					$html_vars[] = (object) array(
						'ID'         => $user['ID'],
						'user_email' => $user['user_email'],
						'first_name' => $user['first_name'],
						'last_name'  => $user['last_name'],
						'quiz_score' => $score,
						'quiz_modal' => $modal_link,
						'quiz_date'  => $date
					);
				}


			}
		}

		return $html_vars;
	}

	public function report_scripts() {
		global $post;

		if ( Utilities::has_shortcode( $post, 'uo_groups_quiz_report' ) || Utilities::has_block( $post, 'uncanny-learndash-groups/uo-groups-quiz-report' ) ) {

			wp_enqueue_script( 'ulgm-datatables', Utilities::get_vendor( 'datatables/js/jquery.dataTables.min.js' ), array( 'jquery' ), Utilities::get_version(), true );

			wp_enqueue_script( 'ulgm-datatables-responsive', Utilities::get_vendor( 'datatables/js/dataTables.responsive.min.js' ), array(
				'jquery',
				'ulgm-datatables'
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-datatables-buttons', Utilities::get_vendor( 'datatables/js/dataTables.buttons.min.js' ), array(
				'jquery',
				'ulgm-datatables'
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-datatables-buttons-html5', Utilities::get_vendor( 'datatables/js/buttons.html5.min.js' ), array(
				'jquery',
				'ulgm-datatables',
				'ulgm-datatables-buttons'
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-datatables-buttons-flash', Utilities::get_vendor( 'datatables/js/buttons.flash.min.js' ), array(
				'jquery',
				'ulgm-datatables',
				'ulgm-datatables-buttons'
			), Utilities::get_version(), true );

			// Front End Questionnaire JS
			wp_register_script( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.js' ), array(
				'jquery',
				'ulgm-datatables'
			), Utilities::get_version(), true );

			$group_drop_downs = $this->get_groups_drop_downs();

			// Attach API data to custom-toolkit-js // TODO change path
			$api_setup = array(
				'root'               => esc_url_raw( rest_url() . $this->root_path ),
				'nonce'              => \wp_create_nonce( 'wp_rest' ),
				'currentUser'        => get_current_user_id(),
				'localized'          => $this->get_frontend_localized_strings(),
				'groupQuizzes'       => $group_drop_downs['quizzes_objects'],
				'courseGroupQuizzes' => $group_drop_downs['course_quizzes_objects'],
				'relationships'      => $group_drop_downs['relationships']
			);

			wp_localize_script( 'ulgm-frontend', 'groupQuizReportSetup', $api_setup );

			wp_enqueue_script( 'ulgm-frontend' );

			wp_enqueue_style( 'ulgm-datatables-styles', Utilities::get_vendor( 'datatables/css/jquery.dataTables.min.css' ), [], Utilities::get_version() );
			wp_enqueue_style( 'ulgm-datatables-responsive', Utilities::get_vendor( 'datatables/css/responsive.dataTables.min.css' ), [], Utilities::get_version() );
			wp_enqueue_style( 'ulgm-font-awesome', Utilities::get_vendor( 'fontawesome/css/font-awesome.min.css' ), array(), Utilities::get_version() );
			wp_enqueue_script( 'ulgm-select2', Utilities::get_vendor( 'select2/js/select2.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_style( 'ulgm-select2', Utilities::get_vendor( 'select2/css/select2.min.css' ), array(), Utilities::get_version() );
			// Load styles
			wp_register_style( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.css' ), array(), Utilities::get_version() );
			$user_colors = Utilities::user_colors();
			wp_add_inline_style( 'ulgm-frontend', $user_colors );
			wp_enqueue_style( 'ulgm-frontend', $user_colors );

		}
	}


	private function get_frontend_localized_strings() {

		$localized_strings = array();

		$localized_strings['username'] = __( 'Username', 'uncanny-learndash-groups' );

		$localized_strings['userEmail'] = __( 'Email', 'uncanny-learndash-groups' );

		$localized_strings['firstName'] = __( 'First Name', 'uncanny-learndash-groups' );

		$localized_strings['lastName'] = __( 'Last Name', 'uncanny-learndash-groups' );

		$localized_strings['quizScore'] = __( 'Quiz Score', 'uncanny-learndash-groups' );

		$localized_strings['detailedReport'] = __( 'Detailed Report', 'uncanny-learndash-groups' );

		$localized_strings['date'] = __( 'Date', 'uncanny-learndash-groups' );

		$localized_strings['csvExport'] = __( 'CSV Export', 'uncanny-learndash-groups' );

		$localized_strings['selectCourse'] = sprintf( __( 'Select %s', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'course' ) );

		$localized_strings['noCourse'] = __( 'No courses Available', 'uncanny-learndash-groups' );

		$localized_strings['selectUser'] = __( 'Select User', 'uncanny-learndash-groups' );

		$localized_strings['noUsers'] = __( 'No Users Available', 'uncanny-learndash-groups' );

		$localized_strings['lengthMenu'] = sprintf( _x( 'Show %s entries', '%s is a number', 'uncanny-learndash-groups' ), '_MENU_' );

		$localized_strings['zeroRecords'] = __( 'No matching records found', 'uncanny-learndash-groups' );

		$localized_strings['info'] = sprintf( _x( 'Showing page %1$s of %2$s', 'Both %1$s and %2$s are numbers', 'uncanny-learndash-groups' ), '_PAGE_', '_PAGES_' );

		$localized_strings['infoEmpty'] = __( 'Showing 0 to 0 of 0 entries', 'uncanny-learndash-groups' );

		$localized_strings['infoFiltered'] = sprintf( _x( '(filtered from %s total entries)', '%s is a number', 'uncanny-learndash-groups' ), '_MAX_' );

		$localized_strings['loadingRecords'] = __( 'Loading', 'uncanny-learndash-groups' );

		$localized_strings['sSearch'] = __( 'Search', 'uncanny-learndash-groups' );

		$localized_strings['all'] = __( 'All', 'uncanny-learndash-groups' );

		$localized_strings['selectQuiz'] = sprintf( __( 'Select %s', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'quiz' ) );

		$localized_strings = apply_filters( 'quiz-report-table-strings', $localized_strings );


		return $localized_strings;
	}
}