<?php

namespace uncanny_ceu;

// If this file is called directly, abort.
use TCPDF;

if ( ! defined( 'WPINC' ) ) {
	die;
}


/**
 * Class CeuCapabilities
 * @package uncanny_ceu
 */
class AwardCertificate {

	public $save_path;
	public $save_time;

	/**
	 * class constructor
	 *
	 */
	function __construct() {

		$this->save_path = WP_CONTENT_DIR . '/uploads/ceu-certs/';
		if ( ! file_exists( $this->save_path ) ) {
			mkdir( $this->save_path, 0755, true );
		}

		add_action( 'template_redirect', array( $this, 'template_redirect_access' ), 9, 0 );

		add_action( 'uo_course_completion_time', array( $this, 'uo_course_completion_time_func' ), 10, 3 );
		add_action( 'learndash_course_completed', array( $this, 'schedule_learndash_before_course_completed' ), 30, 1 );
		add_action( 'uo_ceu_scheduled_learndash_course_completed', array(
			$this,
			'learndash_before_course_completed'
		), 30, 2 );

	}

	/**
	 * @param $atts
	 */
	public function schedule_learndash_before_course_completed( $atts ) {
		$pass_args = [
			$atts['user']->ID,
			$atts['course']->ID,
		];

		wp_schedule_single_event( time() + 45, 'uo_ceu_scheduled_learndash_course_completed', $pass_args );
	}

	/**
	 * @param $time
	 * @param $course_id
	 * @param $user_id
	 */
	function uo_course_completion_time_func( $time, $course_id, $user_id ) {
		if ( ! empty( $time ) ) {
			$this->save_time = $time;
		} else {
			$this->save_time = ! empty( get_user_meta( $user_id, 'course_completed_' . $course_id, true ) ) ? get_user_meta( $user_id, 'course_completed_' . $course_id, true ) : time();
		}
	}


	/**
	 * @param $user_id
	 * @param $current_course_id
	 */
	function learndash_before_course_completed( $user_id, $current_course_id ) {
		$current_user         = new \WP_User( $user_id );
		$current_course_title = get_the_title( $current_course_id );
		$ceu_value            = get_post_meta( $current_course_id, 'ceu_value', true );

		// No ceu value for this course, stop here
		if ( empty( $ceu_value ) ) {
			return;
		}

		// Get course completed on time
		if ( empty( $this->save_time ) ) {
			$completion_date = time();
		} else {
			$completion_date = $this->save_time;
		}
		$current_time = get_user_meta( $user_id, 'course_completed_' . $current_course_id, true );

		$completion_date = apply_filters( 'uo_course_completion_time', $completion_date, $current_course_id, $current_user->ID );
		//Fallback
		if ( empty( $completion_date ) ) {
			$completion_date = ! empty( $current_time ) ? $current_time : time();
		}
		
		// Store course completion CEU data
		update_user_meta( $current_user->ID, 'ceu_earned_' . $completion_date . '_' . $current_course_id, $ceu_value );
		update_user_meta( $current_user->ID, 'ceu_date_' . $completion_date . '_' . $current_course_id, $completion_date );
		update_user_meta( $current_user->ID, 'ceu_title_' . $completion_date . '_' . $current_course_id, $current_course_title );
		update_user_meta( $current_user->ID, 'ceu_course_' . $completion_date . '_' . $current_course_id, $current_course_id );

		// Get Certificate Triggers
		global $wpdb;

		$query                    = "SELECT post_id, meta_key, meta_value FROM $wpdb->postmeta WHERE meta_key = 'ceu_cert_after_earned' OR meta_key = 'ceu_cert_after_course_completions'";
		$ceu_certificate_triggers = $wpdb->get_results( $query );

		// If there are no trigger move on
		if ( empty( $ceu_certificate_triggers ) ) {
			return;
		}

		// Create a workable data
		$ceu_certificate_triggers = $this->normalize_triggers( $ceu_certificate_triggers );

		// Maybe Trigger certificate for Course Completions
		$this->maybe_trigger_course_completions_certificate( $current_user, $current_course_id, $completion_date, $ceu_certificate_triggers );

		// Maybe Trigger certificate for Course Completions
		$this->maybe_trigger_ceu_earned_certificate( $current_user, $current_course_id, $completion_date, $ceu_certificate_triggers );


	}

	/**
	 * @param $current_user
	 * @param $current_course_id
	 * @param $date
	 * @param $ceu_certificate_triggers
	 */
	function maybe_trigger_course_completions_certificate( $current_user, $current_course_id, $date, $ceu_certificate_triggers ) {

		$completed_course_ids = array();

		$user_meta = get_user_meta( $current_user->ID );

		// Loop through all meta to find possible courses completed
		foreach ( $user_meta as $meta_key => $meta_value ) {

			// Catch any earned credit
			if ( false !== strpos( $meta_key, 'ceu_course_' ) ) {

				$course_id = absint( $meta_value[0] );

				// CEU value did not validate as a real number, most likely the course meta was not set
				if ( ! $course_id ) {
					continue;
				}

				$completed_course_ids[] = $course_id;

			}
		}

		foreach ( $ceu_certificate_triggers['ceu_cert_after_course_completions'] as $certificate_id => $trigger ) {

			/*
			 * $trigger is all course ids in an array that need to be completed
			 * $completed_course_ids is all the course in an array to have been completed
			 *
			 * array instersect returns all the course ids that match between the courses that need to be compeleted and one that are completed
			 * if the amount of matched course ids are equal to the amount of $triggers(course that need to be completed) are eqal then the met the criteria
			 */
			if ( count( $trigger ) === count( array_intersect( $trigger, $completed_course_ids ) ) ) {

				$ceu_certificate_earned = get_user_meta( $current_user->ID, 'ceu_cert_earned_' . $certificate_id, true );

				// Certificate was not earned before, lets award it
				if ( '' === $ceu_certificate_earned ) {

					// Store time certificate was earned on
					update_user_meta( $current_user->ID, 'ceu_cert_earned_' . $certificate_id, $date );

					if ( 'yes' === get_option( 'uncanny-ceu-credits-cert-enabled', 'no' ) ) {
						$args = $this->setup_variables( $certificate_id, $current_user, $current_course_id );
						$cert = $this->generate_pdf( $args );
						$this->send_cert_email( $current_user, $cert, $current_course_id );
					}
				}
			}
		}
	}

	/**
	 * @param $current_user
	 * @param $current_course_id
	 * @param $date
	 * @param $ceu_certificate_triggers
	 */
	function maybe_trigger_ceu_earned_certificate( $current_user, $current_course_id, $date, $ceu_certificate_triggers ) {

		$user_meta = get_user_meta( $current_user->ID );

		$total_earned = 0;

		// Loop through all meta to find possible credits earned
		foreach ( $user_meta as $meta_key => $meta_value ) {

			// Catch any earned credit
			if ( false !== strpos( $meta_key, 'ceu_earned_' ) ) {

				$ceu_earned = (float) $meta_value[0];

				// CEU value did not validate as a real number, most likely the course meta was not set
				if ( ! $ceu_earned ) {
					continue;
				}

				$total_earned += $ceu_earned;

			}
		}

		foreach ( $ceu_certificate_triggers['ceu_cert_after_earned'] as $certificate_id => $trigger ) {

			$ceu_needed = (float) $trigger;

			// Amount of CEUs is not set correctly
			if ( ! (float) $ceu_needed ) {
				continue;
			}

			if ( $total_earned >= $ceu_needed ) {

				$ceu_certificate_earned = get_user_meta( $current_user->ID, 'ceu_cert_earned_' . $certificate_id, true );

				// Certificate was not earned before, lets award it
				if ( '' === $ceu_certificate_earned ) {

					// Store time certificate was earned on
					update_user_meta( $current_user->ID, 'ceu_cert_earned_' . $certificate_id, $date );

					if ( 'yes' === get_option( 'uncanny-ceu-credits-cert-enabled', 'no' ) ) {
						$args = $this->setup_variables( $certificate_id, $current_user, $current_course_id );
						$cert = $this->generate_pdf( $args );
						$this->send_cert_email( $current_user, $cert, $current_course_id );
					}
				}
			}
		}
	}

	/**
	 * Set Post ID as object key
	 *
	 * @param array $triggers
	 *
	 * @return array $new_ceu_cetificate_triggers
	 */
	private function normalize_triggers( $triggers ) {

		$new_triggers = array();

		// trigger based on course completions
		$new_triggers['ceu_cert_after_course_completions'] = array();

		// trigger based on CEUs earned
		$new_triggers['ceu_cert_after_earned'] = array();

		foreach ( $triggers as $trigger ) {

			if ( 'ceu_cert_after_course_completions' === $trigger->meta_key ) {

				$course_ids                                                             = array_map( 'floatval', json_decode( $trigger->meta_value ) );
				$new_triggers['ceu_cert_after_course_completions'][ $trigger->post_id ] = $course_ids;
			}

			if ( 'ceu_cert_after_earned' === $trigger->meta_key ) {
				$new_triggers['ceu_cert_after_earned'][ $trigger->post_id ] = (float) $trigger->meta_value;
			}

		}

		return $new_triggers;

	}

	/**
	 *  Add certificate redirect
	 *
	 */
	function template_redirect_access() {

		if ( isset( $_GET['ceu_access_key'] ) && ! empty( $_GET['ceu_access_key'] ) ) {
			$access_key = $_GET['ceu_access_key'];
		} else {
			// This is not a CEU url, move on wih WordPress
			return;
		}

		global $post;


		/**
		 * Added check to ensure $post is not empty
		 *
		 * @since 2.3.0.3
		 */
		if ( empty( $post ) ) {
			return;
		}

		if ( ! ( $post instanceof \WP_Post ) ) {
			return;
		}


		$post_type = $post->post_type;
		$post_id   = $post->ID;

		if ( empty( $post_type ) || empty( $post_id ) ) {
			return;
		}

		if ( $post_type == 'sfwd-certificates' ) {

			$access_vars = SharedFunctions::decrypt( $access_key, wp_salt() );

			$access_vars = explode( '-', $access_vars );

			// Did the access key provide a User ID
			if ( isset( $access_vars[0] ) && absint( $access_vars[0] ) ) {
				$user_id = absint( $access_vars[0] );
			} else {
				// User ID was not in access key
				$this->set_404();

				return;

			}

			$user = get_user_by( 'ID', $user_id );

			// User does not exist anymore
			if ( ! $user ) {
				$this->set_404();

				return;
			}

			// Did the access key provide an award type
			if ( isset( $access_vars[1] ) && $access_vars[1] === "ceu_cert_earned" ) {
				$award = $access_vars[1];
			} else {
				// Award type was not in access key
				$this->set_404();

				return;
			}

			$earned_certificate = '';

			if ( 'ceu_cert_earned' === $award ) {
				$earned_certificate = get_user_meta( $user->ID, 'ceu_cert_earned_' . $post_id, true );
			}

			// User has not earned the certificate
			if ( empty( $earned_certificate ) ) {
				$this->set_404();

				return;
			}


			/**
			 * Use LD Certificate to PDF creation to genderate PDF
			 */

			// Set set the user that the PDF will be generated for
			wp_set_current_user( $user->ID );

			// Load conversion code
			require_once( LEARNDASH_LMS_PLUGIN_DIR . 'includes/ld-convert-post-pdf.php' );

			// Set header so we don't generate a 404
			status_header( 200 );

			// Create the PDF
			post2pdf_conv_post_to_pdf();

			// ALL GOOD :)
			die();


		} else {
			// Access Key what user on a page that wasn't a certificate post type
			$this->set_404();

			return;
		}


	}

	/**
	 * If the url does not validate, redirect to WP 404 page
	 */
	private function set_404() {
		global $wp_query;
		$wp_query->set_404();
		status_header( 404 );
	}

	/**
	 * Send email notification of earned certificate
	 *
	 * @param object $current_user
	 * @param int $certificate
	 * @param int $courses
	 */
	function send_cert_email( $current_user, $certificate, $courses = 0 ) {

		$course_names = array();

		if ( $certificate ) {
			if ( empty( get_option( 'uncanny-ceu-multiple-certs-subject', '' ) ) ) {
				$email_subject = __( 'You earned a Certificate', 'uncanny-ceu' );
			} else {
				$email_subject = get_option( 'uncanny-ceu-multiple-certs-subject' );
			}

			if ( empty( get_option( 'uncanny-ceu-multiple-certs-subject', '' ) ) ) {
				$email_message = __( "Congratulations, you earned a certificate!\r\nYour certificate is attached to this email.", 'uncanny-ceu' );
			} else {
				$email_message = get_option( 'uncanny-ceu-multiple-certs-message' );
			}

			if ( ! empty( $courses ) && is_array( $courses ) ) {
				foreach ( $courses as $l_id ) {
					$course_names[] = get_the_title( $l_id );
				}
			} else {
				$course_names = array( get_the_title( $courses ) );
			}
			$email_message = str_ireplace( '%User%', $current_user->display_name, $email_message );
			$email_message = str_ireplace( '%User First Name%', $current_user->first_name, $email_message );
			$email_message = str_ireplace( '%User Last Name%', $current_user->last_name, $email_message );
			$email_message = str_ireplace( '%User Email%', $current_user->user_email, $email_message );
			$email_message = str_ireplace( '%Courses%', join( ', ', $course_names ), $email_message );

			$email_subject = str_ireplace( '%User%', $current_user->display_name, $email_subject );
			$email_subject = str_ireplace( '%User First Name%', $current_user->first_name, $email_subject );
			$email_subject = str_ireplace( '%User Last Name%', $current_user->last_name, $email_subject );
			$email_subject = str_ireplace( '%User Email%', $current_user->user_email, $email_subject );
			$email_subject = str_ireplace( '%Group Name%', '', $email_subject );
			$email_subject = str_ireplace( '%Courses%', join( ', ', $course_names ), $email_subject );

//			Utilities::log( $current_user, '$current_user', true, 'cert' );
//			Utilities::log( $certificate, '$certificate', true, 'cert' );
//			Utilities::log( $courses, '$courses', true, 'cert' );

			//Sending email to user first
			wp_mail( $current_user->user_email, $email_subject, $email_message, array(), $certificate );

			//Now let's see if we want to send email to admin and group leader
			$is_admin       = get_option( 'uncanny-ceu-multiple-notify-admin', 'no' );
			$is_group_admin = get_option( 'uncanny-ceu-multiple-notify-group-leader', 'no' );

			$email_subject = str_ireplace( 'You earned', $current_user->first_name . ' earned', $email_subject );

			if ( 'yes' === $is_admin ) {
				wp_mail( get_bloginfo( 'admin_email' ), $email_subject, $email_message, array(), $certificate );
			}
			if ( 'yes' === $is_group_admin ) {
				$get_leaders = array();
				$user_groups = learndash_get_users_group_ids( $current_user->ID, true );
				if ( ! empty( $user_groups ) ) {
					foreach ( $user_groups as $group ) {
						$has_group_leader = learndash_get_groups_administrators( $group, true );

						if ( ! empty( $has_group_leader ) ) {
							foreach ( $has_group_leader as $leader ) {
								if ( learndash_is_group_leader_of_user( $leader->ID, $current_user->ID ) ) {
									$ll                      = get_user_by( 'ID', $leader->ID );
									$get_leaders[ $group ][] = $ll->user_email;
								}
							}
						}
					}
				}

				//self::trace_logs( $get_leaders, 'Get Leaders', 'pdf' );
				if ( ! empty( $get_leaders ) ) {
					foreach ( $get_leaders as $key => $value ) {
						$email_message = str_ireplace( '%Group Name%', get_the_title( $key ), $email_message );
						$email_subject = str_ireplace( '%Group Name%', get_the_title( $key ), $email_subject );
						//$email     = join( ', ', $value );
						wp_mail( $value, $email_subject, $email_message, array(), $certificate );

					}
				}

				//Store sent certificate to user meta and/or if do not is active, delete file from server
				foreach ( $certificate as $cert_key => $p_certs ) {
					if ( 'no' === get_option( 'uncanny-ceu-multiple-do-not-store-certs', 'no' ) ) {
						update_user_meta( $current_user->ID, 'uo-ceu-cert-' . $cert_key, $p_certs );
					} else {
						if ( file_exists( $p_certs ) ) {
							unlink( $p_certs );
						}
					}
				}
			}
		}
	}

	/**
	 * @param $certificate_id
	 * @param $current_user
	 * @param $course_id
	 *
	 * @return array
	 */
	function setup_variables( $certificate_id, $current_user, $course_id ) {
		$cert_name = "$certificate_id-" . sanitize_title( $current_user->user_email ) . '-' . current_time('timestamp');
		$args      = array(
			'certificate_post' => $certificate_id,
			'save_path'        => $this->save_path,
			'user'             => $current_user,
			'file_name'        => $cert_name,
			'parameters'       => array(
				'userID'            => $current_user->ID,
				'course-id'         => $course_id,
				'course-name'       => get_the_title( $course_id ),
				'print-certificate' => 1,
			),
		);

		return $args;
	}

	/**
	 * @param $args
	 *
	 * @return string
	 */
	function generate_pdf( $args ) {
		$certificate_id  = $args['certificate_post'];
		$save_path       = $args['save_path'];
		$file_name       = $args['file_name'];
		$user            = $args['user'];
		$parameters      = $args['parameters'];
		$post_id         = intval( $certificate_id );
		$post_data       = get_post( $post_id );
		$monospaced_font = '';
		$l               = '';
		$config_lang     = 'eng';
		$ratio           = 1.25;
		$title           = strip_tags( $post_data->post_title );
		//$content             = $post_data->post_content;
		$target_post_id      = $post_id;
		$get_by_http_request = 0;
		$shortcode           = 'parse';

		ob_start();

		//$title = $post_data->post_title;
		// For qTranslate
		if ( function_exists( 'qtrans_use' ) && ! empty( $this->q_config['language'] ) ) {
			$title = qtrans_use( $this->q_config['language'], $title, false );
		}

		$title = strip_tags( $title );

		$permalink   = get_permalink( $post_data->ID );
		$author_data = get_userdata( $post_data->post_author );

		if ( $author_data->display_name ) {
			$author = $author_data->display_name;
		} else {
			$author = $author_data->user_nicename;
		}

		$tag       = array();
		$tags      = '';
		$tags_data = wp_get_post_tags( $post_data->ID );

		if ( $tags_data ) {
			foreach ( $tags_data as $val ) {
				$tag[] = $val->name;
			}
			$tags = implode( ' ', $tag );
		}

		if ( $get_by_http_request == 1 ) {
			$permalink_url = get_permalink( $post_id );
			$response_data = wp_remote_get( $permalink_url );
			$content       = preg_replace( '|^.*?<!-- post2pdf-converter-begin -->(.*?)<!-- post2pdf-converter-end -->.*?$|is', '$1', $response_data['body'] );
		} else {
			$content = $post_data->post_content;

			// For qTranslate
			if ( function_exists( 'qtrans_use' ) && ! empty( $this->q_config['language'] ) ) {
				$content = qtrans_use( $this->q_config['language'], $content, true );
			}
		}

		if ( ! empty( $_GET['lang'] ) ) {
			$config_lang = substr( esc_html( $_GET['lang'] ), 0, 3 );
		}

		if ( ! empty( $_GET['file'] ) ) {
			$filename_type = $_GET['file'];
		}

		if ( $filename_type == 'title' && $target_post_id == 0 ) {
			$filename = $post_data->post_title;

			// For qTranslate
			if ( function_exists( 'qtrans_use' ) && ! empty( $this->q_config['language'] ) ) {
				$filename = qtrans_use( $this->q_config['language'], $filename, false );
			}
		} else {
			$filename = $post_id;
		}

		$filename = substr( $filename, 0, 255 );

		$chached_filename = '';

		if ( $target_post_id != 0 ) {
			$filename = WP_CONTENT_DIR . '/tcpdf-pdf/' . $filename;
		}

		// For qTranslate
		if ( function_exists( 'qtrans_use' ) && ! empty( $this->q_config['language'] ) ) {
			$filename = $filename . '_' . $this->q_config['language'];
		}

		if ( ! empty( $_GET['font'] ) ) {
			$font = esc_html( $_GET['font'] );
		}

		if ( ! empty( $_GET['monospaced'] ) ) {
			$monospaced_font = esc_html( $_GET['monospaced'] );
		}

		if ( ! empty( $_GET['fontsize'] ) ) {
			$font_size = intval( $_GET['fontsize'] );
		}

		if ( ! empty( $_GET['subsetting'] ) && ( $_GET['subsetting'] == 1 || $_GET['subsetting'] == 0 ) ) {
			$subsetting_enable = $_GET['subsetting'];
		}

		if ( $subsetting_enable == 1 ) {
			$subsetting = 'true';
		} else {
			$subsetting = 'false';
		}

		if ( ! empty( $_GET['ratio'] ) ) {
			$ratio = floatval( $_GET['ratio'] );
		}

		if ( ! empty( $_GET['header'] ) ) {
			$header_enable = $_GET['header'];
		}

		if ( ! empty( $_GET['logo'] ) ) {
			$logo_enable = $_GET['logo'];
		}

		if ( ! empty( $_GET['logo_file'] ) ) {
			$logo_file = esc_html( $_GET['logo_file'] );
		}

		if ( ! empty( $_GET['logo_width'] ) ) {
			$logo_width = intval( $_GET['logo_width'] );
		}

		if ( ! empty( $_GET['wrap_title'] ) ) {
			$wrap_title = $_GET['wrap_title'];
		}

		if ( ! empty( $_GET['footer'] ) ) {
			$footer_enable = $_GET['footer'];
		}

		if ( ! empty( $_GET['filters'] ) ) {
			$filters = $_GET['filters'];
		}

		if ( ! empty( $_GET['shortcode'] ) ) {
			$shortcode = esc_html( $_GET['shortcode'] );
		}

		if ( $target_post_id != 0 ) {
			$destination = 'F';
		} else {
			$destination = 'I';
		}

		$content = preg_replace( '/(\[courseinfo)/', '[courseinfo user_id="' . $user->ID . '" course_id="' . $parameters['course-id'] . '" ', $content );
		$content = preg_replace( '/(\[usermeta)/', '[usermeta user_id="' . $user->ID . '" ', $content );
		$content = apply_filters( 'uo_generate_course_certificate_content', $content, $user->ID, $parameters['course-id'] );

		// Delete shortcode for POST2PDF Converter
		$content = preg_replace( '|\[pdf[^\]]*?\].*?\[/pdf\]|i', '', $content );

		// For WP-Syntax, WP-CodeBox(GeSHi) and WP-GeSHi-Highlight -- syntax highlighting with clean, small and valid (X)HTML
		if ( function_exists( 'wp_syntax_highlight' ) || function_exists( 'wp_codebox_before_filter' ) || function_exists( 'wp_geshi_main' ) ) {
			$content = preg_replace_callback( "/<pre[^>]*?lang=['\"][^>]*?>(.*?)<\/pre>/is", array(
				$this,
				post2pdf_conv_sourcecode_wrap_pre_and_esc
			), $content );
		}

		// For CodeColorer(GeSHi)
		if ( class_exists( 'CodeColorerLoader' ) ) {
			$content = preg_replace_callback( "/<code[^>]*?lang=['\"][^>]*?>(.*?)<\/code>/is", array(
				$this,
				post2pdf_conv_sourcecode_wrap_pre_and_esc
			), $content );
		}

		// For WP Code Highlight
		if ( function_exists( 'wp_code_highlight_filter' ) ) {
			$content = wp_code_highlight_filter( $content );
			$content = preg_replace( '/<pre[^>]*?>(.*?)<\/pre>/is', '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );
		}

		// Parse shortcode before applied WP default filters
		if ( $shortcode == 'parse' && $get_by_http_request != 1 ) {

			// For WP SyntaxHighlighter
			if ( function_exists( 'wp_sh_add_extra_bracket' ) ) {
				$content = wp_sh_add_extra_bracket( $content );
			}

			if ( function_exists( 'wp_sh_do_shortcode' ) ) {
				$content = wp_sh_do_shortcode( $content );
			}

			// For SyntaxHighlighter Evolved
			if ( class_exists( 'SyntaxHighlighter' ) ) {
				global $SyntaxHighlighter;
				if ( method_exists( 'SyntaxHighlighter', 'parse_shortcodes' ) && method_exists( 'SyntaxHighlighter', 'shortcode_hack' ) ) {
					$content = $SyntaxHighlighter->parse_shortcodes( $content );
				}
			}

			// For SyntaxHighlighterPro
			if ( class_exists( 'GoogleSyntaxHighlighterPro' ) ) {
				global $googleSyntaxHighlighter;
				if ( method_exists( 'GoogleSyntaxHighlighterPro', 'bbcode' ) ) {
					$content = $googleSyntaxHighlighter->bbcode( $content );
				}
			}

			// For CodeColorer(GeSHi)
			if ( class_exists( 'CodeColorerLoader' ) ) {
				$content = preg_replace_callback( "/\[cc[^\]]*?lang=['\"][^\]]*?\](.*?)\[\/cc\]/is", array(
					$this,
					post2pdf_conv_sourcecode_wrap_pre_and_esc
				), $content );
			}
		} else if ( $get_by_http_request != 1 ) {

			// For WP SyntaxHighlighter
			if ( function_exists( 'wp_sh_strip_shortcodes' ) ) {
				$content = wp_sh_strip_shortcodes( $content );
			}

			// For SyntaxHighlighterPro
			if ( class_exists( 'GoogleSyntaxHighlighterPro' ) ) {
				global $googleSyntaxHighlighter;
				if ( method_exists( 'GoogleSyntaxHighlighterPro', 'bbcode_strip' ) ) {
					$content = $googleSyntaxHighlighter->bbcode_strip( $content );
				}
			}

			// For CodeColorer(GeSHi)
			if ( class_exists( 'CodeColorerLoader' ) ) {
				$content = preg_replace_callback( "/\[cc[^\]]*?lang=['\"][^\]]*?\](.*?)\[\/cc\]/is", array(
					$this,
					post2pdf_conv_sourcecode_esc
				), $content );
			}
		}

		// Apply WordPress default filters to title and content
		if ( $filters == 1 && $get_by_http_request != 1 ) {

			if ( has_filter( 'the_title', 'wptexturize' ) ) {
				$title = wptexturize( $title );
			}

			if ( has_filter( 'the_title', 'convert_chars' ) ) {
				$title = convert_chars( $title );
			}

			if ( has_filter( 'the_title', 'trim' ) ) {
				$title = trim( $title );
			}

			if ( has_filter( 'the_title', 'capital_P_dangit' ) ) {
				$title = capital_P_dangit( $title );
			}

			if ( has_filter( 'the_content', 'wptexturize' ) ) {
				$content = wptexturize( $content );
			}

			if ( has_filter( 'the_content', 'convert_smilies' ) ) {
				$content = convert_smilies( $content );
			}

			if ( has_filter( 'the_content', 'convert_chars' ) ) {
				$content = convert_chars( $content );
			}

			if ( has_filter( 'the_content', 'wpautop' ) ) {
				$content = wpautop( $content );
			}

			if ( has_filter( 'the_content', 'shortcode_unautop' ) ) {
				$content = shortcode_unautop( $content );
			}

			if ( has_filter( 'the_content', 'prepend_attachment' ) ) {
				$content = prepend_attachment( $content );
			}

			if ( has_filter( 'the_content', 'capital_P_dangit' ) ) {
				$content = capital_P_dangit( $content );
			}
		}
		//$content = do_shortcode( $content );

		$dir = $this->get_learndash_plugin_directory();
		if ( $dir ) {
			// Include TCPDF
			require_once $dir . 'includes/vendor/tcpdf/config/lang/' . $config_lang . '.php';
			require_once $dir . 'includes/vendor/tcpdf/tcpdf.php';
		} else {
			return false;
		}

		$certificate_details = get_post_meta( $certificate_id, 'learndash_certificate_options', true );

		if ( $certificate_details ) {
			$page_size        = $certificate_details['pdf_page_format'];
			$page_orientation = $certificate_details['pdf_page_orientation'];
		} else {
			$page_size        = 'LETTER';
			$page_orientation = 'L';
		}

		// Create a new object
		$pdf = new TCPDF( $page_orientation, PDF_UNIT, $page_size, true, 'UTF-8', false, false );

		// Set document information
		$pdf->SetCreator( PDF_CREATOR );
		$pdf->SetAuthor( get_bloginfo( 'name' ) );
		$pdf->SetTitle( $title . '_' . $post_id . '_' . get_bloginfo( 'name' ) );
		// Set default monospaced font
		$pdf->SetDefaultMonospacedFont( $monospaced_font );

		// Set header data
		if ( mb_strlen( $title, 'UTF-8' ) < 42 ) {
			$header_title = $title;
		} else {
			$header_title = mb_substr( $title, 0, 42, 'UTF-8' ) . '...';
		}

		if ( $header_enable == 1 ) {
			if ( $logo_enable == 1 && $logo_file ) {
				$pdf->SetHeaderData( $logo_file, $logo_width, $header_title, 'by ' . $author . ' - ' . $permalink );
			} else {
				$pdf->SetHeaderData( '', 0, $header_title, 'by ' . $author . ' - ' . $permalink );
			}
		}

		// Set header and footer fonts
		if ( $header_enable == 1 ) {
			$pdf->setHeaderFont( Array( $font, '', PDF_FONT_SIZE_MAIN ) );
		}

		if ( $footer_enable == 1 ) {
			$pdf->setFooterFont( Array( $font, '', PDF_FONT_SIZE_DATA ) );
		}

		// Remove header/footer
		if ( $header_enable == 0 ) {
			$pdf->setPrintHeader( false );
		}

		if ( $header_enable == 0 ) {
			$pdf->setPrintFooter( false );
		}
		// Set margins
		$pdf->SetMargins( PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT );

		if ( $header_enable == 1 ) {
			$pdf->SetHeaderMargin( PDF_MARGIN_HEADER );
		}

		if ( $footer_enable == 1 ) {
			$pdf->SetFooterMargin( PDF_MARGIN_FOOTER );
		}

		// Set auto page breaks
		$pdf->SetAutoPageBreak( true, PDF_MARGIN_BOTTOM );

		// Set image scale factor
		$pdf->setImageScale( $ratio );

		// Set some language-dependent strings
		$pdf->setLanguageArray( $l );

		// Set fontsubsetting mode
		$pdf->setFontSubsetting( $subsetting );

		// Set font
		$pdf->SetFont( $font, '', $font_size, true );

		// Add a page
		$pdf->AddPage();

		// Create post content to print
		if ( $wrap_title == 1 ) {
			if ( mb_strlen( $title, 'UTF-8' ) < 33 ) {
				$title = $title;
			} else {
				$title = mb_substr( $title, 0, 33, 'UTF-8' ) . '<br />' . mb_substr( $title, 33, 222, 'UTF-8' );
			}
		}

		// Parse shortcode after applied WP default filters
		if ( $shortcode == 'parse' && $get_by_http_request != 1 ) {

			// For WP QuickLaTeX
			if ( function_exists( 'quicklatex_parser' ) ) {
				$content = quicklatex_parser( $content );
			}

			// For WP shortcode API
			$content = do_shortcode( $content );
		} else if ( $get_by_http_request != 1 ) {

			// For WP shortcode API
			$content = strip_shortcodes( $content );
		}

		// Convert relative image path to absolute image path
		$content = preg_replace( "/<img([^>]*?)src=['\"]((?!(http:\/\/|https:\/\/|\/))[^'\"]+?)['\"]([^>]*?)>/i", '<img$1src="' . site_url() . '/$2"$4>', $content );

		// Set image align to center
		$content = preg_replace_callback( "/(<img[^>]*?class=['\"][^'\"]*?aligncenter[^'\"]*?['\"][^>]*?>)/i", 'post2pdf_conv_image_align_center', $content );

		// Add width and height into image tag
		$content = preg_replace_callback( "/(<img[^>]*?src=['\"]((http:\/\/|https:\/\/|\/)[^'\"]*?(jpg|jpeg|gif|png))['\"])([^>]*?>)/i", 'post2pdf_conv_img_size', $content );

		// For WP QuickLaTeX
		if ( function_exists( 'quicklatex_parser' ) ) {
			$content = preg_replace_callback( '/(<p class="ql-(center|left|right)-displayed-equation" style="line-height: )([0-9]+?)(px;)(">)/i', array(
				$this,
				post2pdf_conv_qlatex_displayed_equation
			), $content );
			$content = str_replace( '<p class="ql-center-picture">', '<p class="ql-center-picture" style="text-align: center;"><span class="ql-right-eqno"> &nbsp; <\/span><span class="ql-left-eqno"> &nbsp; <\/span>', $content );
		}

		// For common SyntaxHighlighter
		$content = preg_replace( "/<pre[^>]*?class=['\"][^'\"]*?brush:[^'\"]*?['\"][^>]*?>(.*?)<\/pre>/is", '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );
		$content = preg_replace( "/<script[^>]*?type=['\"]syntaxhighlighter['\"][^>]*?>(.*?)<\/script>/is", '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );
		$content = preg_replace( "/<pre[^>]*?name=['\"]code['\"][^>]*?>(.*?)<\/pre>/is", '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );
		$content = preg_replace( "/<textarea[^>]*?name=['\"]code['\"][^>]*?>(.*?)<\/textarea>/is", '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );
		$content = preg_replace( '/\n/', '<br/>', $content ); //"\n" should be treated as a next line

		// For WP-SynHighlight(GeSHi)
		if ( function_exists( 'wp_synhighlight_settings' ) ) {
			$content = preg_replace( "/<pre[^>]*?class=['\"][^>]*?>(.*?)<\/pre>/is", '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );
			$content = preg_replace( '|<div[^>]*?class="wp-synhighlighter-outer"><div[^>]*?class="wp-synhighlighter-expanded"><table[^>]*?><tr><td[^>]*?><a[^>]*?></a><a[^>]*?class="wp-synhighlighter-title"[^>]*?>[^<]*?</a></td><td[^>]*?><a[^>]*?><img[^>]*?/></a>[^<]*?<a[^>]*?><img[^>]*?/></a>[^<]*?<a[^>]*?><img[^>]*?/></a>[^<]*?</td></tr></table></div>|is', '', $content );
		}

		// For other sourcecode
		$content = preg_replace( '/<pre[^>]*?><code[^>]*?>(.*?)<\/code><\/pre>/is', '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $content );

		// For blockquote
		$content = preg_replace( '/<blockquote[^>]*?>(.*?)<\/blockquote>/is', '<blockquote style="color: #406040;">$1</blockquote>', $content );

		// Combine title with content
		$formatted_title = '<h1 style="text-align:center;">' . $title . '</h1>';

		//$formatted_post = $formatted_title . '<br/><br/>' . $content;    (Title will not appear on PDF)
		$formatted_post = '<br/><br/>' . $content;
		$formatted_post = preg_replace( '/(<[^>]*?font-family[^:]*?:)([^;]*?;[^>]*?>)/is', '$1' . $font . ',$2', $formatted_post );

		// get featured image
		$postid   = get_the_id(); //Get current post id
		$img_file = $this->learndash_get_thumb_path( $certificate_id ); //The same function from theme's[twentytwelve here] function.php

		//Only print image if it exists
		if ( $img_file != '' ) {

			//Print BG image
			$pdf->setPrintHeader( false );

			// get the current page break margin
			$bMargin = $pdf->getBreakMargin();

			// get current auto-page-break mode
			$auto_page_break = $pdf->getAutoPageBreak();

			// disable auto-page-break
			$pdf->SetAutoPageBreak( false, 0 );

			// Get width and height of page for dynamic adjustments
			$pageH = $pdf->getPageHeight();
			$pageW = $pdf->getPageWidth();

			//Print the Background
			$pdf->Image( $img_file, $x = '0', $y = '0', $w = $pageW, $h = $pageH, $type = '', $link = '', $align = '', $resize = false, $dpi = 300, $palign = '', $ismask = false, $imgmask = false, $border = 0, $fitbox = false, $hidden = false, $fitonpage = false, $alt = false, $altimgs = array() );

			// restore auto-page-break status
			$pdf->SetAutoPageBreak( $auto_page_break, $bMargin );

			// set the starting point for the page content
			$pdf->setPageMark();
		}

		// Print post
		$pdf->writeHTMLCell( $w = 0, $h = 0, $x = '', $y = '', $formatted_post, $border = 0, $ln = 1, $fill = 0, $reseth = true, $align = '', $autopadding = true );

		// Set background
		$pdf->SetFillColor( 255, 255, 127 );
		$pdf->setCellPaddings( 0, 0, 0, 0 );
		// Print signature

		ob_clean();
		// Output pdf document
		$full_path = $save_path . $file_name . '.pdf';

		$pdf->Output( $full_path, 'F' ); /* F means saving on server. */

		return $full_path;

	}

	/**
	 * @param $this ->post_id
	 *
	 * @return string
	 */
	private function learndash_get_thumb_path( $post_id ) {
		$thumbnail_id = get_post_meta( $post_id, '_thumbnail_id', true );
		if ( $thumbnail_id ) {
			$img_path      = get_post_meta( $thumbnail_id, '_wp_attached_file', true );
			$upload_url    = wp_upload_dir();
			$img_full_path = $upload_url['basedir'] . '/' . $img_path;

			return $img_full_path;
		}
	}

	/**
	 * @return string
	 */
	private function get_learndash_plugin_directory() {
		if ( defined( LEARNDASH_LMS_PLUGIN_DIR ) ) {
			return LEARNDASH_LMS_PLUGIN_DIR;
		} else {
			$all_plugins = get_plugins();
			$dir         = '';
			if ( $all_plugins ) {
				foreach ( $all_plugins as $key => $plugin ) {
					if ( 'LearnDash LMS' === $plugin['Name'] ) {
						$dir = plugin_dir_path( $key );

						return WP_PLUGIN_DIR . '/' . $dir;
						break;
					}
				}
			}

			return $dir;
		}
	}
}