<?php
/**
 * Plugin Name: LearnDash - Slack
 * Plugin URI: https://realbigplugins.com/plugins/learndash-slack/
 * Description: Slack Integration for LearnDash
 * Version: 1.2.4
 * Text Domain: learndash-slack
 * Author: Real Big Plugins
 * Author URI: https://realbigplugins.com
 * Contributors: d4mation, joelworsham
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'LearnDash_Slack' ) ) {

	/**
	 * Main LearnDash_Slack class
	 *
	 * @since		1.0.0
	 */
	final class LearnDash_Slack {
		
		/**
		 * @var			array $plugin_data Holds Plugin Header Info
		 * @since		1.0.0
		 */
		public $plugin_data;
		
		/**
		 * @var			LearnDash_Slack_License_Field $license_field Adding our License Field to the LearnDash Licensing Page
		 * @since		1.0.0
		 */
		public $license_field;
		
		/**
		 * @var			LearnDash_Slack_Admin $admin Admin Settings
		 * @since		1.0.0
		 */
		public $admin;
		
		/**
		 * @var			LearnDash_Slack_Notification_Handler $notification_handler Notifications System
		 * @since		1.0.0
		 */
		public $notification_handler;
		
		/**
		 * @var			LearnDash_Slack_Notification_Integration Integrates into our Notification System. Serves as an example on how to utilize it.
		 * @since		1.0.0
		 */
		public $notification_integration;
		
		/**
		 * @var			LearnDash_Slack_Notification_Triggers $notification_triggers Notification Triggers. Serves as an example on how to Trigger Notifications
		 * @since		1.0.0
		 */
		public $notification_triggers;
		
		/**
		 * @var			LearnDash_Slack_API $slack_api Slack API
		 * @since		1.0.0
		 */
		public $slack_api;
		
		/**
		 * @var			array $admin_notices Stores all our Admin Notices to fire at once
		 * @since		1.0.0
		 */
		private $admin_notices;
		
		/**
		 * @var			array $integration_errors Stores all our Integration Errors to fire at once
		 * @since		1.2.0
		 */
		private $integration_errors;
		
		/**
		 * @var			RBP_Support $support RBP Support module
		 * @since		1.2.0
		 */
		public $support;

		/**
		 * Get active instance
		 *
		 * @access		public
		 * @since		1.0.0
		 * @return		object self::$instance The one true LearnDash_Slack
		 */
		public static function instance() {
			
			static $instance = null;
			
			if ( null === $instance ) {
				$instance = new static();
			}
			
			return $instance;

		}
		
		protected function __construct() {
			
			$this->setup_constants();
			$this->load_textdomain();
			
			global $wp_settings_errors;
			
			// That's a descriptive class name! /s
			if ( ! class_exists( 'Semper_Fi_Module' ) ) {
				
				$this->admin_notices[] = sprintf( _x( '%s requires %s to be installed!', 'Missing Plugin Dependency Error', 'learndash-slack' ), '<strong>' . $this->plugin_data['Name'] . '</strong>', '<a href="//www.learndash.com/" target="_blank"><strong>LearnDash</strong></a>' );
				
				if ( ! has_action( 'admin_notices', array( $this, 'admin_notices' ) ) ) {
					add_action( 'admin_notices', array( $this, 'admin_notices' ) );
				}
				
				return;
				
			}
			
			if ( defined( 'LEARNDASH_VERSION' ) 
				&& ( version_compare( LEARNDASH_VERSION, '2.2.1.2' ) < 0 ) ) {
				
				$this->admin_notices[] = sprintf( _x( '%s requires v%s of %s or higher to be installed!', 'Outdated Dependency Error', 'learndash-slack' ), '<strong>' . $this->plugin_data['Name'] . '</strong>', '2.2.1.2', '<a href="//www.learndash.com/" target="_blank"><strong>LearnDash</strong></a>' );
				
				if ( ! has_action( 'admin_notices', array( $this, 'admin_notices' ) ) ) {
					add_action( 'admin_notices', array( $this, 'admin_notices' ) );
				}
				
			}
			
			$this->require_necessities();
			
			// Register our CSS/JS for the whole plugin
			add_action( 'init', array( $this, 'register_scripts' ) );
			
		}

		/**
		 * Setup plugin constants
		 *
		 * @access		private
		 * @since		1.0.0
		 * @return		void
		 */
		private function setup_constants() {
			
			// WP Loads things so weird. I really want this function.
			if ( ! function_exists( 'get_plugin_data' ) ) {
				require_once ABSPATH . '/wp-admin/includes/plugin.php';
			}
			
			// Only call this once, accessible always
			$this->plugin_data = get_plugin_data( __FILE__ );

			if ( ! defined( 'LearnDash_Slack_VER' ) ) {
				// Plugin version
				define( 'LearnDash_Slack_VER', $this->plugin_data['Version'] );
			}

			if ( ! defined( 'LearnDash_Slack_DIR' ) ) {
				// Plugin path
				define( 'LearnDash_Slack_DIR', plugin_dir_path( __FILE__ ) );
			}

			if ( ! defined( 'LearnDash_Slack_URL' ) ) {
				// Plugin URL
				define( 'LearnDash_Slack_URL', plugin_dir_url( __FILE__ ) );
			}
			
			if ( ! defined( 'LearnDash_Slack_FILE' ) ) {
				// Plugin File
				define( 'LearnDash_Slack_FILE', __FILE__ );
			}

		}

		/**
		 * Internationalization
		 *
		 * @access		private 
		 * @since		1.0.0
		 * @return		void
		 */
		private function load_textdomain() {

			// Set filter for language directory
			$lang_dir = LearnDash_Slack_DIR . '/languages/';
			$lang_dir = apply_filters( 'LearnDash_Slack_languages_directory', $lang_dir );

			// Traditional WordPress plugin locale filter
			$locale = apply_filters( 'plugin_locale', get_locale(), 'learndash-slack' );
			$mofile = sprintf( '%1$s-%2$s.mo', 'learndash-slack', $locale );

			// Setup paths to current locale file
			$mofile_local   = $lang_dir . $mofile;
			$mofile_global  = WP_LANG_DIR . '/learndash-slack/' . $mofile;

			if ( file_exists( $mofile_global ) ) {
				// Look in global /wp-content/languages/learndash-slack/ folder
				// This way translations can be overridden via the Theme/Child Theme
				load_textdomain( 'learndash-slack', $mofile_global );
			}
			else if ( file_exists( $mofile_local ) ) {
				// Look in local /wp-content/plugins/learndash-slack/languages/ folder
				load_textdomain( 'learndash-slack', $mofile_local );
			}
			else {
				// Load the default language files
				load_plugin_textdomain( 'learndash-slack', false, $lang_dir );
			}

		}
		
		/**
		 * Include different aspects of the Plugin
		 * 
		 * @access		private
		 * @since		1.0.0
		 * @return		void
		 */
		private function require_necessities() {
			
			if ( is_admin() ) {
				
				// Legacy LD Menu API
				if ( version_compare( LEARNDASH_VERSION, '2.4' ) < 0 ) {
					
					require_once LearnDash_Slack_DIR . '/core/admin/settings-tab/class-learndash-slack-legacy-settings-tab.php';
					
				}
				else {
				
					require_once LearnDash_Slack_DIR . '/core/admin/settings-tab/class-learndash-slack-settings-tab.php';
					
				}
				
				require_once LearnDash_Slack_DIR . '/core/admin/class-learndash-slack-admin.php';
				$this->admin = new LearnDash_Slack_Admin();
				
			}
			
			// Change Prefix for RBP Support Object
			add_filter( 'rbp_support_prefix', array( $this, 'change_rbp_support_prefix' ) );
			
			// Allow RBP Support Form to be a proper <form>
			add_filter( 'ld_slack_support_form_tag', array( $this, 'change_rbp_support_form_tag' ) );
			
			// Support Module
			require_once __DIR__ . '/includes/rbp-support/rbp-support.php';
			$this->support = new RBP_Support( LearnDash_Slack_FILE, array(
				'support_form' => array(
					'enabled' => array(
						'title' => _x( 'Need some help with %s?', '%s is the Plugin Name', 'learndash-slack' ),
						'subject_label' => __( 'Subject', 'learndash-slack' ),
						'message_label' => __( 'Message', 'learndash-slack' ),
						'send_button' => __( 'Send', 'learndash-slack' ),
						'subscribe_text' => _x( 'We make other cool plugins and share updates and special offers to anyone who %ssubscribes here%s.', 'Both %s are used to place HTML for the <a> in the message', 'learndash-slack' ),
						'validationError' => _x( 'This field is required', 'Only used by legacy browsers for JavaScript Form Validation', 'learndash-slack' ),
						'success' => __( 'Support message succesfully sent!', 'learndash-slack' ),
						'error' => __( 'Could not send support message.', 'learndash-slack' ),
					),
					'disabled' => array(
						'title' => _x( 'Need some help with %s?', '%s is the Plugin Name', 'learndash-slack' ),
						'disabled_message' => __( 'Premium support is disabled. Please register your product and activate your license for this website to enable.', 'learndash-slack' )
					),
				),
				'licensing_fields' => array(
					'title' => _x( '%s License', '%s is the Plugin Name', 'learndash-slack' ),
					'deactivate_button' => __( 'Deactivate', 'learndash-slack' ),
					'activate_button' => __( 'Activate', 'learndash-slack' ),
					'delete_deactivate_button' => __( 'Delete and Deactivate', 'learndash-slack' ),
					'delete_button' => __( 'Delete', 'learndash-slack' ),
					'license_active_label' => __( 'License Active', 'learndash-slack' ),
					'license_inactive_label' => __( 'License Inactive', 'learndash-slack' ),
					'save_activate_button' => __( 'Save and Activate', 'learndash-slack' ),
				),
				'license_nag' => array(
					'register_message' => _x( 'Register your copy of %s now to receive automatic updates and support.', '%s is the Plugin Name', 'learndash-slack' ),
					'purchase_message' => _x( 'If you do not have a license key, you can %1$spurchase one%2$s.', 'Both %s are used to place HTML for the <a> in the message', 'learndash-slack' ),
				),
				'license_activation' => _x( '%s license successfully activated.', '%s is the Plugin Name', 'learndash-slack' ),
				'license_deletion' => _x( '%s license successfully deleted.', '%s is the Plugin Name', 'learndash-slack' ),
				'license_deactivation' => array(
					'error' => _x( 'Error: could not deactivate the license for %s', '%s is the Plugin Name', 'learndash-slack' ),
					'success' => _x( '%s license successfully deactivated.', '%s is the Plugin Name', 'learndash-slack' ),
				),
				'license_error_messages' => array(
					'expired' => _x( 'Your license key expired on %s.', '%s is a localized timestamp', 'learndash-slack' ),
					'revoked' => __( 'Your license key has been disabled.', 'learndash-slack' ),
					'missing' => __( 'Invalid license.', 'learndash-slack' ),
					'site_inactive' => __( 'Your license is not active for this URL.', 'learndash-slack' ),
					'item_name_mismatch' => _x( 'This appears to be an invalid license key for %s.', '%s is the Plugin Name', 'learndash-slack' ),
					'no_activations_left' => __( 'Your license key has reached its activation limit.', 'learndash-slack' ),
					'no_connection' => _x( '%s cannot communicate with %s for License Key Validation. Please check your server configuration settings.', '%s is the Plugin Name followed by the Store URL', 'learndash-slack' ),
					'default' => __( 'An error occurred, please try again.', 'learndash-slack' ),
				),
				'beta_checkbox' => array(
					'label' => __( 'Enable Beta Releases', 'learndash-slack' ),
					'disclaimer' => __( 'Beta Releases should not be considered as Stable. Enabling this on your Production Site is done at your own risk.', 'learndash-slack' ),
					'enabled_message' => _x( 'Beta Releases for %s enabled.', '%s is the Plugin Name', 'learndash-slack' ),
					'disabled_message' => _x( 'Beta Releases for %s disabled.', '%s is the Plugin Name', 'learndash-slack' ),
				),
			) );
			
			// Revert this change so that it won't harm any future potential instances of the object
			remove_filter( 'rbp_support_prefix', array( $this, 'change_rbp_support_prefix' ) );
			
			require_once LearnDash_Slack_DIR . '/core/licensing/class-learndash-slack-licensing-field.php';
			$this->license_field = new LearnDash_Slack_License_Field();
			
			require_once LearnDash_Slack_DIR . '/core/slack/class-learndash-slack-api.php';
			$this->slack_api = new LearnDash_Slack_API();
			
			require_once LearnDash_Slack_DIR . '/core/notifications/class-learndash-slack-notification-handler.php';
			$this->notification_handler = new LearnDash_Slack_Notification_Handler();
			
			require_once LearnDash_Slack_DIR . '/core/slack/class-learndash-slack-notification-integration.php';
			$this->notification_integration = new LearnDash_Slack_Notification_Integration();
			
			require_once LearnDash_Slack_DIR . '/core/slack/class-learndash-slack-notification-triggers.php';
			$this->notification_triggers = new LearnDash_Slack_Notification_Triggers();
			
			// Includes functionality that only exists if Groups are created
			require_once LearnDash_Slack_DIR . '/core/integrations/ld-groups/class-ld-slack-ld-groups.php';
			
			// If EDD Software Licensing is Active
			if ( class_exists( 'LearnDash_Gradebook' ) ) {
				
				if ( defined( 'LEARNDASH_GRADEBOOK_VERSION' ) &&
				  version_compare( LEARNDASH_GRADEBOOK_VERSION, '1.2.0' ) >= 0 ) {
					
					require_once LearnDash_Slack_DIR . '/core/integrations/ld-gradebook/class-ld-slack-ld-gradebook.php';
					
				}
				else {
					
					$this->integration_errors[] = sprintf( _x( '%s includes features which integrate with %s, but v%s or greater of %s is required.', 'Outdated Integration Error', 'learndash-slack' ), '<strong>' . $this->plugin_data['Name'] . '</strong>', '<a href="' . admin_url( 'update-core.php' ) . '"><strong>LearnDash - Gradebook</strong></a>', '1.2.0', '<a href="' . admin_url( 'update-core.php' ) . '"><strong>LearnDash - Gradebook</strong></a>' );
					
				}
				
			}
			
			// Output all Integration-related Errors just above the Notificiation Repeater
			if ( ! empty( $this->integration_errors ) ) {
				
				add_action( 'ld_slack_before_repeater', array( $this, 'integration_errors' ) );
				
			}
			
			if ( is_admin() ) {
				
				// Don't try to run upgrade routines unless we're on the Backend
				// Less hitting the DB for a version number as well as if there are noticies to show, it will be more apparent as to what needs done by the User
				// If they have hit the Backend via updating within WP this will check right away
				// If they do it via SFTP, it won't until the next time they load the Dashboard
				// Since I don't want literally every interaction within WP to check the Database for this, this is how it will be for now
				
				add_action( 'init', function() {
				
					// We must wait until init so that Post Types and Taxonomies are created
					$this->upgrade_routine();
					
				} );
				
				
			}
			
		}
		
		/**
		 * We are going to alter the Prefix to match what it was before the Support Module was included
		 * In this particular case, the Support Module handled Licensing the same way as LD Slack, so we don't need an upgrade script!
		 * 
		 * @param		string $prefix RBP_Support Prefix
		 *                                    
		 * @access		public
		 * @since		1.2.0
		 * @return		string RBP_Support Prefix
		 */
		public function change_rbp_support_prefix( $prefix ) {
			
			return 'ld_slack';
			
		}
		
		/**
		 * Change RBP Support Form to use a proper <form> and not utilize the special JavaScript Validation normally used
		 * 
		 * @param		string $tag RBP_Support Form Tag
		 *                                      
		 * @access		public
		 * @since		1.2.0
		 * @return		string RBP_Support Form Tag
		 */
		public function change_rbp_support_form_tag( $tag ) {
			
			return 'form';
			
		}
		
		/**
		 * Grab LearnDash Slack Fields
		 * 
		 * $param		$query Whether to run all the Queries for Choices or not
		 * 
		 * @access		public
		 * @since		1.0.0
		 * @return		array  LearnDash_Slack Settings API Fields
		 */
		public function get_settings_fields( $query = true ) {

			$courses_array = array();
			$lessons_array = array();
			$topics_array = array();
			$quizzes_array = array();
			$webhooks_array = array();
			$global_defaults = array();

			// Only run through all these queries when we need them
			if ( $query ) {
				
				// Holds all non-repeater values
				$global_defaults = get_option( 'learndash_slack' );

				$base_args = array(
					'posts_per_page' => -1,
					'orderby' => 'title',
					'order' => 'ASC',
				);

				$courses = get_posts( array(
					'post_type' => 'sfwd-courses',
				) + $base_args );

				foreach ( $courses as $course ) {
					$courses_array[ $course->ID ] = $course->post_title;
				}

				$lessons = get_posts( array(
					'post_type' => 'sfwd-lessons',
				) + $base_args );

				foreach ( $lessons as $lesson ) {
					$lessons_array[ $lesson->ID ] = $lesson->post_title;
				}

				$topics = get_posts( array(
					'post_type' => 'sfwd-topic',
				) + $base_args );

				foreach ( $topics as $topic ) {
					$topics_array[ $topic->ID ] = $topic->post_title;
				}

				$quizzes = get_posts( array( 
					'post_type' => 'sfwd-quiz',
				) + $base_args );

				foreach ( $quizzes as $quiz ) {
					$quizzes_array[ $quiz->ID ] = $quiz->post_title;
				}
				
				$webhooks = get_terms( array(
					'taxonomy' => 'ld-slack-rbm-webhook',
					'hide_empty' => false,
					'order' => 'DESC',
					'orderby' => 'date',
				) );

				foreach ( $webhooks as $webhook ) {

					$webhooks_array[ $webhook->term_id ] = $webhook->name;

				}
				
				if ( isset( $global_defaults['webhook_default'] ) ) {
					unset( $webhooks_array[ $global_defaults['webhook_default'] ] );
				}

			}
			
			$fields = array( 
				'slack_post_id' => array(
					'type'  => 'hook',
				),
				'admin_title' => array(
					'label' => _x( 'Identifier for this Notification', 'Admin Title Field Label', 'learndash-slack' ),
					'type'  => 'text',
					'classes' => array(
						'slack-post-title',
					),
					'placeholder' => __( 'New Slack Notification', 'learndash-slack' ),
					'tooltip_title' => __( 'Indentifier for this Notification', 'learndash-slack' ),
					'tooltip_desc'  => __( 'Helps distinguish Notifications from one another on the Settings Screen. If left blank, your Notification will be labeled &ldquo;New Slack Notification&rdquo;.', 'learndash-slack' ),
				),
				'trigger' => array(
					'type' => 'select',
					'label' => _x( 'Slack Trigger', 'Slack Trigger Label', 'learndash-slack' ),
					'chosen' => true,
					'classes' => array(
						'ld-slack-trigger',
						'required',
					),
					'choices' => array(
						'' => _x( '-- Select a Slack Trigger --', 'Slack Trigger Default', 'learndash-slack' ),
					) + $this->get_slack_triggers(),
					'std' => '',
				),
				'course' => array(
					'type' => 'select',
					'label' => LearnDash_Custom_Label::get_label( 'course' ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
					'classes' => array(
						'ld-slack-course',
						'ld-slack-conditional',
						'complete_course',
						'enroll_course',
						'course_expires',
						'required',
					),
					'choices' => array(
						'all' => sprintf( _x( 'All %s', 'All Items in a Select Field', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'courses' ) ),
					) + $courses_array,
					'std' => '',
				),
				'exclude_course' => array(
					'type' => 'select',
					'label' => sprintf( __( 'Exclude %s (Optional)', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
					'classes' => array(
						'ld-slack-exclude-select', // This is a conditional field, but not in the same way as others. It will be hidden/shown based on the value the preceding field
						'ld-slack-exclude-parent-course', // This is used to determine the conditional field parent
					),
					'choices' => $courses_array,
					'std' => '',
				),
				'lesson' => array(
					'type' => 'select',
					'label' => LearnDash_Custom_Label::get_label( 'lesson' ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
					'classes' => array(
						'ld-slack-lesson',
						'ld-slack-conditional',
						'complete_lesson',
						'lesson_available',
						'required',
					),
					'choices' => array(
						'all' => sprintf( _x( 'All %s', 'All Items in a Select Field', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lessons' ) ),
					) + $lessons_array,
					'std' => '',
				),
				'exclude_lesson' => array(
					'type' => 'select',
					'label' => sprintf( __( 'Exclude %s (Optional)', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
					'classes' => array(
						'ld-slack-exclude-select', // This is a conditional field, but not in the same way as others. It will be hidden/shown based on the value the preceding field
						'ld-slack-exclude-parent-lesson', // This is used to determine the conditional field parent
					),
					'choices' => $lessons_array,
					'std' => '',
				),
				'topic' => array(
					'type' => 'select',
					'label' => LearnDash_Custom_Label::get_label( 'topic' ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
					'classes' => array(
						'ld-slack-topic',
						'ld-slack-conditional',
						'complete_topic',
						'required',
					),
					'choices' => array(
						'all' => sprintf( _x( 'All %s', 'All Items in a Select Field', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topics' ) ),
					) + $topics_array,
					'std' => '',
				),
				'exclude_topic' => array(
					'type' => 'select',
					'label' => sprintf( __( 'Exclude %s (Optional)', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
					'classes' => array(
						'ld-slack-exclude-select', // This is a conditional field, but not in the same way as others. It will be hidden/shown based on the value the preceding field
						'ld-slack-exclude-parent-topic', // This is used to determine the conditional field parent
					),
					'choices' => $topics_array,
					'std' => '',
				),
				'quiz' => array(
					'type' => 'select',
					'label' => LearnDash_Custom_Label::get_label( 'quiz' ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
					'classes' => array( 
						'ld-slack-quiz',
						'ld-slack-conditional',
						'pass_quiz',
						'fail_quiz',
						'complete_quiz',
						'required',
					),
					'choices' => array(
						'all' => sprintf( _x( 'All %s', 'All Items in a Select Field', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quizzes' ) ),
					) + $quizzes_array,
					'std' => '',
				),
				'exclude_quiz' => array(
					'type' => 'select',
					'label' => sprintf( __( 'Exclude %s (Optional)', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
					'chosen' => true,
					'multiple' => true,
					'placeholder' => sprintf( _x( '-- Select %s --', 'Select Field Default', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
					'classes' => array(
						'ld-slack-exclude-select', // This is a conditional field, but not in the same way as others. It will be hidden/shown based on the value the preceding field
						'ld-slack-exclude-parent-quiz', // This is used to determine the conditional field parent
					),
					'choices' => $quizzes_array,
					'std' => '',
				),
				'before_days' => array(
					'type' => 'number',
					'label' => _x( 'How many days before?', 'How many days before? Field Label', 'learndash-slack' ),
					'classes' => array(
						'ld-slack-conditional',
						'course_expires',
						'required',
					),
				),
				'after_days' => array(
					'type' => 'number',
					'label' => _x( 'After how many days?', 'After how many days? Field Label', 'learndash-slack' ),
					'classes' => array(
						'ld-slack-conditional',
						'not_logged_in',
						'required',
					),
				),
				'replacement_hints' => array(
					'type' => 'hook',
				),
				'message_pretext' => array(
					'type' => 'text',
					'label' => _x( 'Message Pre-Text (Optional)', 'Message Pre-Text Field Label', 'learndash-slack' ),
					'tooltip_title' => __( 'Message Pre-text', 'learndash-slack' ),
					'tooltip_desc'  => __( 'Shows directly below Username and above the Title/Message.', 'learndash-slack' ),
				),
				'message_title' => array(
					'type' => 'text',
					'classes' => array(
						'ld-slack-title'
					),
					'label' => _x( 'Message Title (Optional)', 'Message Title Field Label', 'learndash-slack' ),
					'placeholder' => get_the_title(),
					'tooltip_title' => __( 'Message Title', 'learndash-slack' ),
					'tooltip_desc'  => __( 'If left blank this will default to the Notification Identifier.', 'learndash-slack' ),
				),
				'message_text' => array(
					'type' => 'textarea',
					'label' => _x( 'Message (Optional)', 'Message Text Field Label', 'learndash-slack' ),
					'min_rows' => 3,
				),
				'webhook_url' => array(
					'type' => 'select',
					'classes' => 'ld-slack-webhook-url',
					'label' => _x( 'Slack Webhook URL (Optional)', 'Slack Webhook URL Field Label', 'learndash-slack' ),
					'std' => '',
					'tooltip_title' => __( 'Slack Webhook URL', 'learndash-slack' ),
					'tooltip_desc'  => __( 'This overrides the Default Webhook URL. This can be useful if you want a specific Notification to send to another Slack Team entirely.', 'learndash-slack' ),
					'choices' => array( 
						'' => __( '-- Default Webhook URL --', 'learndash-slack' ),
					) + $webhooks_array,
					'chosen' => true,
				),
				'channel' => array(
					'type' => 'text',
					'label' => _x( 'Slack Channel (Optional)', 'Slack Channel Field Label', 'learndash-slack' ),
					'placeholder' => _x( 'Webhook Default', 'Webhook Default Placeholder Text', 'learndash-slack' ),
					'tooltip_title' => __( 'Slack Channel', 'learndash-slack' ),
					'tooltip_desc'  => __( 'This overrides the Default Channel defined by the Webhook URL. Notifications can be sent to individual Users instead by entering their Username like so: <code>@&laquo;username&raquo;</code>.', 'learndash-slack' ),
				),
				'username' => array(
					'type' => 'text',
					'label' => _x( 'Username (Optional)', 'Username Field Label', 'learndash-slack' ),
					'placeholder' => get_bloginfo( 'name' ),
					'tooltip_title' => __( 'Username', 'learndash-slack' ),
					'tooltip_desc'  => sprintf( __( 'This controls who the Notification appears to be from. It does not have to be a valid User in your Slack Team. This will default to &ldquo;%s&rdquo;.', 'learndash-slack' ), get_bloginfo( 'name' ) ),
				),
				'icon' => array(
					'type' => 'text',
					'label' => _x( 'Icon Emoji or Image URL (Optional)', 'Icon Emoji or Image URL Field Label', 'learndash-slack' ),
					'placeholder' => _x( 'Webhook Default', 'Webhook Default Placeholder Text', 'learndash-slack' ),
					'tooltip_title' => __( 'Icon Emoji or Image URL', 'learndash-slack' ),
					'tooltip_desc'  => __( 'This accepts an Emoji (Example: <code>:rocket:</code>) or an Image URL. You can even use Custom Emojis defined within your Slack Team. If left empty, it will default to the Emoji or Image defined by the Webhook URL.', 'learndash-slack' ),
				),
				'color' => array(
					'type' => 'color',
					'label' => _x( 'Color', 'Color Field Label', 'learndash-slack' ),
					'tooltip_title' => __( 'Color', 'learndash-slack' ),
					'tooltip_desc'  => __( 'Shows next to Message Title and Message.', 'learndash-slack' ),
				),
			);

			return apply_filters( 'ld_slack_settings_fields', $fields );

		}
		
		/**
		 * Returns Array of LearnDash Slack Triggers
		 * 
		 * @access		public
		 * @since		1.2.0
		 * @return		array LearnDash Slack Triggers
		 */
		public function get_slack_triggers() {
			
			$triggers = apply_filters( 'ld_slack_triggers', array(
				'enroll_course' => sprintf( _x( 'User enrolls into a %s', 'Course Enrollment Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
				'complete_course' => sprintf( _x( 'User completes a %s', 'Course Completion Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
				'complete_lesson' => sprintf( _x( 'User completes a %s', 'Lesson Completion Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
				'lesson_available' => sprintf( _x( 'A scheduled %s is available to a User', 'Scheduled Lesson Available Trigger', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'lesson' ) ),
				'complete_topic' => sprintf( _x( 'User completes a %s', 'Topic Completion Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'topic' ) ),
				'pass_quiz' => sprintf( _x( 'User passes a %s', 'Quiz Passed Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
				'fail_quiz' => sprintf( _x( 'User fails a %s', 'Quiz Failed Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
				'complete_quiz' => sprintf( _x( 'User completes a %s', 'Quiz Completion Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'quiz' ) ),
				'upload_assignment' => _x( 'An Assignment is uploaded', 'Assignment Uploaded Trigger Text', 'learndash-slack' ),
				'approve_assignment' => _x( 'An Assignment is approved', 'Assignment Approved Trigger Text', 'learndash-slack' ),
				'not_logged_in' => _x( 'User hasn\'t logged in for "X" days', 'User Hasn\'t Logged In for X Days Trigger Text', 'learndash-slack' ),
				'course_expires' => sprintf( _x( '"X" days before %s expires', 'X Days Before Course Expires Trigger Text', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ) ),
				'essay_graded' => _x( 'Essay question has been put into graded status', 'Essay Graded Trigger Text', 'learndash-slack' ),
			) );
			
			// Ignore Quotation Marks when sorting Alphabetically
			uasort( $triggers, function( $a, $b ) {
				
				return strcmp( trim( $a, '"' ), trim( $b, '"' ) );
				
			} );
			
			return $triggers;
			
		}
		
		/**
		 * Utility Function to insert one Array into another at a specified Index. Useful for the Notification Repeater Field's Filter
		 * @param		array   &$array       Array being modified. This passes by reference.
		 * @param		integer $index        Insertion Index. Even if it is an associative array, give a numeric index. Determine it by doing a foreach() until you hit your desired placement and then break out of the loop.
		 * @param		array   $insert_array Array being Inserted at the Index
		 *                                                           
		 * @access		public
		 * @since		1.2.0
		 * @return		void
		 */
		public function array_insert( &$array, $index, $insert_array ) { 
			
			// First half before the cut-off for the splice
			$first_array = array_splice( $array, 0, $index ); 
			
			// Merge this with the inserted array and the last half of the splice
			$array = array_merge( $first_array, $insert_array, $array );
			
		}
		
		/**
		 * Register our CSS/JS to use later
		 * 
		 * @access		public
		 * @since		1.0.0
		 * @return		void
		 */
		public function register_scripts() {
			
			wp_register_style(
				'learndash-slack-admin',
				LearnDash_Slack_URL . '/assets/css/admin.css',
				null,
				defined( 'WP_DEBUG' ) && WP_DEBUG ? time() : LearnDash_Slack_VER
			);
			
			wp_register_script(
				'learndash-slack-admin',
				LearnDash_Slack_URL . '/assets/js/admin.js',
				array( 'jquery', 'jquery-ui-tooltip', 'jquery-effects-core', 'jquery-effects-highlight', 'wp-color-picker' ),
				defined( 'WP_DEBUG' ) && WP_DEBUG ? time() : LearnDash_Slack_VER,
				true
			);
			
			wp_localize_script( 
				'learndash-slack-admin',
				'ldSlack',
				apply_filters( 'ld_slack_localize_admin_script', array(
					'url' => LearnDash_Slack_URL,
					'chosenWarning' => __( 'Chosen has been loaded by a plugin other than LearnDash Slack. This may cause problems.', 'learndash-slack' ),
				) )
			);
			
		}
		
		/*
		 * Show admin notices.
		 * 
		 * @access		public
		 * @since		1.0.0
		 * @return		HTML
		 */
		public function admin_notices() {
			?>
			<div class="error">
				<?php foreach ( $this->admin_notices as $notice ) : ?>
					<p>
						<?php echo $notice; ?>
					</p>
				<?php endforeach; ?>
			</div>
			<?php
		}
		
		/**
		 * Show Integration errors.
		 * 
		 * @access	  public
		 * @since	  1.2.0
		 * @return	  HTML
		 */
		public function integration_errors() {
			?>
			<div class="integration-error">
				<?php foreach ( $this->integration_errors as $notice ) : ?>
					<p>
						<?php echo $notice; ?>
					</p>
				<?php endforeach; ?>
			</div>
			<?php
		}
		
		/**
		 * Runs on Plugin Activation to set up a WP Cron
		 * 
		 * @access		public
		 * @since		1.0.0
		 * @return		void
		 */
		public static function activate() {
			
			if ( ! wp_next_scheduled( 'ld_slack_daily_cron' ) ) {
				wp_schedule_event( time(), 'daily', 'ld_slack_daily_cron' );
			}
			
		}
		
		/**
		 * Runs on Plugin Deactivation to remove the WP Cron from Activation
		 * 
		 * @access		public
		 * @since		1.0.0
		 * @return		void
		 */
		public static function deactivate() {
			
			wp_clear_scheduled_hook( 'ld_slack_daily_cron' );
			
		}
		
		/**
		 * Run any appropriate Upgrade Routines
		 * If the upgrade requires User Interaction, it forwards the version to the pending_upgrade() method
		 *                                                          
		 * @access		private
		 * @since		1.2.0
		 * @return		bool|void Normally returns nothing, but will bail if appropriate
		 */
		private function upgrade_routine() {
			
			$last_upgrade = get_option( 'ld_slack_last_upgrade', false );
			
			// If nothing is set, assume they've never done an upgrade
			if ( ! $last_upgrade ) {
				$last_upgrade = '1.0.0';
			}
			
			if ( strpos( $last_upgrade, 'pending' !== false ) ) {
				// Pending routines are handled afterward
				// This just skips the need to check other Versions 
			}
			else if ( version_compare( $last_upgrade, '1.2.0' ) < 0 ) {
				
				$global_settings = get_option( 'learndash_slack' );
				$notification_id = apply_filters( 'ld_slack_notification_id', 'rbm' );
				
				// This helps ensure we don't make multiple copies of the same webhook
				$in_use_webhooks = array();
				
				$default_webhook = ( isset( $global_settings['webhook_default'] ) ) ? trim( $global_settings['webhook_default'] ) : '';
				
				// Store any Notification-specific Webhooks as a Term and assign them back to the Notification
				$feeds = get_posts( array(
					'post_type' => "ld-slack-{$notification_id}-feed",
					'numberposts' => -1,
					'order'	=> 'ASC',
				) );
				
				if ( ! empty( $feeds ) && ! is_wp_error( $feeds ) ) {

					foreach ( $feeds as $feed ) {
						
						// Originally this was stored incorrectly
						$webhook = trim( get_post_meta( $feed->ID, "ld_slack_{$notification_id}_feed_webhook", true ) );
						
						if ( ! $webhook ) continue;
						
						$term_id = false;
						if ( $webhook !== $default_webhook && 
							! in_array( $webhook, $in_use_webhooks ) ) {
							
							// Name the Term after the Notification Identifier
							$term = wp_insert_term( trim( get_the_title( $feed->ID ) ), "ld-slack-{$notification_id}-webhook" );
					
							// $term is an Array holding term_id and term_taxonomy_id
							$term_id = $term['term_id'];
							
							$term_meta_id = update_term_meta( $term_id, 'webhook_url', $webhook );
							
							$in_use_webhooks[ $term_id ] = $webhook;
							
						}
						else if ( $webhook !== $default_webhook ) {
							// Term already exists, so let's use that
							$term_id = array_search( $webhook, $in_use_webhooks );
						}
						
						// Correcting the Meta Key
						delete_post_meta( $feed->ID, "ld_slack_{$notification_id}_feed_webhook" );
						
						// No need to store the Term ID if it is the Default
						if ( $webhook !== $default_webhook ) {
							update_post_meta( $feed->ID, "ld_slack_{$notification_id}_feed_webhook_url", $term_id );
						}
						
					}
					
				}
				
				$last_upgrade = '1.2.0';
				
				update_option( 'ld_slack_last_upgrade', $last_upgrade );
				
			}
			
			// Run any Pending routines
			if ( strpos( $last_upgrade, 'pending' ) !== false ) {
				
				$pending_version = str_replace( 'pending-', '', $last_upgrade );
				
				$finished_pending = $this->pending_upgrade( $pending_version );
				
				// We're pending, this needs user interaction before we can proceed
				if ( ! $finished_pending ) {
					return false;
				}
				
			}
			
		}
		
		/**
		 * Check to see if Pending Upgrades have been completed by the User
		 * 
		 * @param		string  $pending_version Upgrade that is Pending
		 *                                                  
		 * @access		private
		 * @since		1.2.0
		 * @return		boolean Whether the Pending Upgrade has been completed or not
		 */
		private function pending_upgrade( $pending_version ) {
			
			return false;
			
		}

	}

} // End Class Exists Check

/**
 * The main function responsible for returning the one true LearnDash_Slack
 * instance to functions everywhere
 *
 * @since		1.0.0
 * @return		\LearnDash_Slack The one true LearnDash_Slack
 */
add_action( 'plugins_loaded', 'LearnDash_Slack_load', 10 );
function LearnDash_Slack_load() {
		
	require_once __DIR__ . '/core/learndash-slack-functions.php';
	LEARNDASHSLACK();

}

register_activation_hook( __FILE__, array( 'LearnDash_Slack', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'LearnDash_Slack', 'deactivate' ) );