<?php
/**
 * Integration for LearnDash Groups
 * While not necessarily an _INTEGRATION_, it doesn't make much sense to show these fields if no Groups exist
 * Having a Required Field where the only option is "All" in these cases would be annoying for the User
 *
 * @since 1.2.0
 *
 * @package LD_Slack
 * @subpackage LD_Slack/core/integrations/ld-groups
 */

defined( 'ABSPATH' ) || die();

final class LD_Slack_LD_Groups {
	
	/**
	 * LD_Slack_LD_Groups constructor.
	 *
	 * @since 1.2.0
	 */
	function __construct() {
			
		// Add new Conditional Fields for the Comment Trigger
		add_filter( 'ld_slack_settings_fields', array( $this, 'add_extra_fields' ) );
		
		// Inject some Checks before we do Replacements or send the Notification
		add_action( 'ld_slack_before_replacements', array( $this, 'before_notification_replacements' ), 10, 5 );
		
		// Add our own Replacement Strings
		add_filter( 'ld_slack_notifications_replacements', array( $this, 'custom_replacement_strings' ), 10, 5 );
		
		// We wait until init here to prevent Issues with LD Gradebook's integration
		add_action( 'init', function() {
		
			// Add our own Hints for the Replacement Strings
			add_filter( 'ld_slack_text_replacement_hints', array( $this, 'custom_replacement_hints' ), 10, 3 );
			
		} );
		
	}
	
	/**
	 * There are a ton of Triggers that utilize Groups. This allows us to grab the full list at once without needing to copy/paste everywhere
	 * 
	 * @access		public
	 * @since		1.2.0
	 * @return		array Groups Triggers
	 */
	public function get_groups_triggers() {
		
		return apply_filters( 'ld_slack_groups_triggers', array(
			'enroll_course',
			'complete_course',
			'complete_lesson',
			'lesson_available',
			'complete_topic',
			'pass_quiz',
			'fail_quiz',
			'complete_quiz',
			'upload_assignment',
			'approve_assignment',
			'not_logged_in',
			'essay_graded',
		) );
		
	}
	
	/**
	 * Conditionally Showing Fields within the Notification Repeater works by adding the Trigger as a HTML Class Name
	 * 
	 * @param	  array $repeater_fields Notification Repeater Fields
	 *												  
	 * @access	  public
	 * @since	  1.2.0
	 * @return	  array Notification Repeater Fields
	 */
	public function add_extra_fields( $repeater_fields ) {
		
		$groups = get_posts( array( 
			'post_type' => 'groups',
			'posts_per_page' => -1,
			'orderby' => 'title',
			'order' => 'ASC',
		) );
		
		if ( empty( $groups ) ) return $repeater_fields;

		$groups_array = array();
		foreach ( $groups as $group ) {
			$groups_array[ $group->ID ] = $group->post_title;
		}
		
		$index = 0;
		foreach ( $repeater_fields as $key => $value ) {
			
			// Find the Numeric Index of the Exclude Quiz Field
			if ( $key == 'exclude_quiz' ) {
				break;
			}
			
			$index++;
			
		}
		
		// Create a new Repeater Field
		$groups = array(
			'group' => array(
				'type' => 'select',
				'label' => __( 'Student is within Group', 'learndash-slack' ),
				'chosen' => true,
				'multiple' => true,
				'placeholder' => _x( '-- Select Group --', 'Select Field Default', 'learndash-slack' ),
				'classes' => array( 
					'ld-slack-group',
					'ld-slack-conditional',
					'required',
				),
				'choices' => array(
					'all' => _x( 'All Groups', 'All Items in a Select Field', 'learndash-slack' ),
				) + $groups_array,
				'std' => '',
			),
		);
		
		// Add Groups Triggers as Classes for Conditionally Showing/Hiding the Field
		$groups['group']['classes'] += $this->get_groups_triggers();
		
		// Insert the new field just after the "Exclude Quiz" Select Field
		LEARNDASHSLACK()->array_insert( $repeater_fields, $index + 1, $groups );
		
		return $repeater_fields;
		
	}
	
	/**
	 * Inject some checks on whether or not to bail on the Notification
	 * 
	 * @param		object $post            WP_Post Object for our Saved Notification Data
	 * @param		array  $fields          Fields used to create the Post Meta
	 * @param		string $trigger         Notification Trigger
	 * @param		string $notification_id ID Used for Notification Hooks
	 * @param		array  &$args           $args Array passed from the original Trigger of the process
	 *                                                                                     
	 * @access		public
	 * @since		1.2.0
	 * @return		void
	 */
	public function before_notification_replacements( $post, $fields, $trigger, $notification_id, &$args ) {
		
		if ( $notification_id == 'rbm' ) {
			
			if ( in_array( $trigger, $this->get_groups_triggers() ) ) {
				
				$groups = get_posts( array( 
					'post_type' => 'groups',
					'posts_per_page' => 1,
				) );

				// Don't bail on the notification, but don't run any checks either
				if ( empty( $groups ) ) return false;
				
				if ( ! is_array( $fields['group'] ) ) $fields['group'] = array( $fields['group'] );
				
				// We are allowing empty here since a Group-related Notification may not have been saved after 1.2.0 of LD Slack was installed with Existing Groups. Empty is treated as the same as "All". Empty is not possible when saving
				if ( ! empty( $fields['group'] ) && 
				   ! in_array( 'all', $fields['group'] ) ) {
					
					// Groups the User is in
					$users_groups = learndash_get_users_group_ids( $args['user_id'] );
					
					// If none of the User's Groups exist in our Notification, bail
					if ( empty( array_intersect( $users_groups, $fields['group'] ) ) ) {
						$args['bail'] = true;
						return false;
					}
					
				}
				
			}
			
		}
		
	}
	
	/**
	 * Based on our Notification ID and Trigger, use some extra Replacement Strings
	 * 
	 * @param		array  $replacements    Notification Fields to check for replacements in
	 * @param		array  $fields          Fields used to create the Post Meta
	 * @param		string $trigger         Notification Trigger
	 * @param		string $notification_id ID used for Notification Hooks
	 * @param		array  $args            $args Array passed from the original Trigger of the process
	 *                                                                                     
	 * @access		public
	 * @since		1.2.0
	 * @return		array  Replaced Strings within each Field
	 */
	public function custom_replacement_strings( $replacements, $fields, $trigger, $notification_id, $args ) {

		if ( $notification_id == 'rbm' ) {

			if ( in_array( $trigger, $this->get_groups_triggers() ) ) {
				
				$groups = get_posts( array( 
					'post_type' => 'groups',
					'posts_per_page' => 1,
				) );

				// Don't run any of the replacements here if there are no Groups
				if ( empty( $groups ) ) return $replacements;
				
				// Groups the User is in
				$users_groups = learndash_get_users_group_ids( $args['user_id'] );
				
				$replacements['%groups%'] = '';
				
				foreach( $users_groups as $group_id ) {
					$replacements['%groups%'] .= "&bull; <" . admin_url( 'post.php?post=' . $group_id . '&action=edit' ) . '|' . get_the_title( $group_id ) . ">\n";
				}
				
			}
			
		}
		
		return $replacements;
		
	}
	
	/**
	 * Add Replacement String Hints
	 * 
	 * @param		array $hints              The main Hints Array
	 * @param		array $user_hints         General Hints for a User. These apply to likely any possible Trigger
	 * @param		array $course_basic_hints The Basic Course Hints. These apply to a lot of different Triggers, so they're available in the Filter easily
	 *                                  
	 * @access		public
	 * @since		1.2.0
	 * @return		array The main Hints Array
	 */
	public function custom_replacement_hints( $hints, $user_hints, $course_basic_hints ) {
		
		$groups = get_posts( array( 
			'post_type' => 'groups',
			'posts_per_page' => 1,
		) );
		
		// Don't add any Replacement Hints if there are no Groups
		if ( empty( $groups ) ) return $hints;
		
		$groups_hints = array(
			'%groups%' => __( 'List of Groups this Student is in.', 'learndash-slack' ),
		);
		
		$group_triggers = $this->get_groups_triggers();
		
		foreach ( $group_triggers as $trigger ) {
			
			$hints[ $trigger ] = $hints[ $trigger ] + $groups_hints;
			
		}
		
		return $hints;
		
	}
	
}

$integrate = new LD_Slack_LD_Groups();