<?php
/**
 * Bootstrapper for the plugin.
 *
 * Makes sure everything is good to go for loading the plugin, and then loads it.
 *
 * @since 1.1.0
 *
 * @package LearnDash_ConvertKitPro
 */

defined( 'ABSPATH' ) || die;

/**
 * Class LearnDash_ConvertKit_Bootstrapper
 *
 * Bootstrapper for the plugin.
 *
 * Makes sure everything is good to go for loading the plugin, and then loads it.
 *
 * @since 1.1.0
 */
class LearnDash_ConvertKit_Bootstrapper {

	/**
	 * Notices to show if cannot load.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @var array
	 */
	private $notices = array();

	/**
	 * LearnDash_ConvertKit_Bootstrapper constructor.
	 *
	 * @since 1.1.0
	 */
	function __construct() {

		add_action( 'plugins_loaded', array( $this, 'maybe_load' ) );
	}

	/**
	 * Maybe loads the plugin.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @global LearnDash_ConvertKit $LearnDash_ConvertKit
	 */
	function maybe_load() {

		$php_version = phpversion();
		$wp_version  = get_bloginfo( 'version' );
		$ld_version  = false;

		// LearnDash required
		if ( ! defined( 'LEARNDASH_VERSION' ) ) {

			$this->notices[] = __( 'LearnDash LMS must be installed and activated.', 'learndash-convertkit' );

		} else {

			$ld_version = LEARNDASH_VERSION;
		}

		// Minimum PHP version
		if ( version_compare( $php_version, '5.3.0' ) === - 1 ) {

			$this->notices[] = sprintf(
				__( 'Minimum PHP version of 5.3.0 required. Current version is %s. Please contact your system administrator to upgrade PHP to its latest version.', 'learndash-convertkit' ),
				$php_version
			);
		}

		// Minimum WordPress version
		if ( version_compare( $wp_version, '4.8.0' ) === - 1 ) {

			$this->notices[] = sprintf(
				__( 'Minimum WordPress version of 4.8.0 required. Current version is %s. Please contact your system administrator to upgrade WordPress to its latest version.', 'learndash-convertkit' ),
				$wp_version
			);
		}

		// Minimum LearnDash version
		if ( $ld_version && version_compare( $ld_version, '2.5.2' ) === - 1 ) {

			$this->notices[] = sprintf(
				__( 'Minimum LearnDash LMS version of 2.5.2 required. Current version is %s. Please contact your system administrator to upgrade LearnDash LMS to its latest version.', 'learndash-convertkit' ),
				$ld_version
			);
		}

		// Don't load and show errors if incompatible environment.
		if ( ! empty( $this->notices ) ) {

			add_action( 'admin_notices', array( $this, 'notices' ) );

			return;
		}

		$this->load();
	}

	/**
	 * Loads the plugin.
	 *
	 * @since 1.1.0
	 * @access private
	 */
	private function load() {

		LearnDash_ConvertKit();
	}

	/**
	 * Shows notices on failure to load.
	 *
	 * @since 1.1.0
	 * @access private
	 */
	function notices() {
		?>
        <div class="notice error">
            <p>
				<?php
				printf(
					__( '%LearnDash - ConvertKit%s could not load because of the following errors:', 'learndash-convertkit' ),
					'<strong>',
					'</strong>'
				);
				?>
            </p>

            <ul>
				<?php foreach ( $this->notices as $notice ) : ?>
                    <li>
                        &bull; <?php echo $notice; ?>
                    </li>
				<?php endforeach; ?>
            </ul>
        </div>
		<?php
	}
}