<?php
/**
 * The Settings API Functions for LearnDash_Slack
 *
 * @since		1.0.0
 *
 * @package LearnDash_Slack
 * @subpackage LearnDash_Slack/core
 */

defined( 'ABSPATH' ) || die();

if ( ! function_exists( 'ld_slack_text_callback' ) ) {

	/**
	 * Creates a Text Field
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_text_callback( $args ) { 

		$args = wp_parse_args( $args, array(
			'id' => '',
			'label' => '',
			'desc' => '',
			'classes' => '',
			'std' => '',
			'placeholder' => '',
			'hidden' => false,
			'input_name' => false,
			'pattern' => false,
			'pattern_message' => false,
			'disabled' => false,
		) );

		if ( ! is_array( $args['classes'] ) ) $args['classes'] = array( $args['classes'] );

		$name = ( $args['input_name'] ) ? $args['input_name'] . '[' . $args['id'] . ']' : $args['id'];
		
		ob_start(); ?>

		<?php if ( ! empty( $args['label'] ) ) : ?>
			<label for="<?php echo esc_attr( $name ); ?>"><?php echo $args['label']; ?></label>
		<?php endif; ?>

		<?php echo do_action( 'ld_slack_after_setting_label', $args ); ?>

		<input type="text" name="<?php echo esc_attr( $name ); ?>" class="ld-slack-field regular-text <?php echo ld_slack_sanitize_html_class( $args['classes'] ); ?>" value="<?php echo esc_attr( $args['std'] ); ?>" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>"<?php echo ( $args['pattern'] ) ? ' pattern="' . $args['pattern'] . '"' : ''; ?><?php echo ( $args['pattern_message'] ) ? ' title="' . $args['pattern_message'] . '"' : ''; ?><?php echo ( $args['disabled'] ) ? ' disabled' : ''; ?> />

		<?php 
		
		/**
		 * This allows more granular inclusion of additional HTML immediately after the actual "input"
		 * 
		 * @since		1.2.0
		 * @return		void
		 */
		do_action( 'ld_slack_after_setting_input_html', $args ); ?>

		<?php if ( ! empty( $args['desc'] ) ) : ?>
			<p class="description">
				<?php echo $args['desc']; ?>
			</p>
		<?php endif;
		
		$html = ob_get_clean();
		
		echo apply_filters( 'ld_slack_after_setting_output', $html, $args );

	}
	
}

if ( ! function_exists( 'ld_slack_textarea_callback' ) ) {

	/**
	 * Creates a Text Field
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_textarea_callback( $args ) { 

		$args = wp_parse_args( $args, array(
			'id' => '',
			'label' => '',
			'desc' => '',
			'classes' => '',
			'std' => '',
			'placeholder' => '',
			'hidden' => false,
			'input_name' => false,
			'rows' => 3,
			'min_rows' => false,
		) );

		if ( ! is_array( $args['classes'] ) ) $args['classes'] = array( $args['classes'] );
		
		if ( $args['min_rows'] ) $args['classes'][] = 'ld-slack-auto-expand-textarea';

		$name = ( $args['input_name'] ) ? $args['input_name'] . '[' . $args['id'] . ']' : $args['id'];
		
		ob_start(); ?>

		<?php if ( ! empty( $args['label'] ) ) : ?>
			<label for="<?php echo esc_attr( $name ); ?>"><?php echo $args['label']; ?></label>
		<?php endif; ?>

		<?php echo do_action( 'ld_slack_after_setting_label', $args ); ?>

		<textarea name="<?php echo esc_attr( $name ); ?>" class="ld-slack-field regular-text <?php echo ld_slack_sanitize_html_class( $args['classes'] ); ?>" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>" rows="<?php echo $args['rows']; ?>"<?php echo ( $args['min_rows'] ) ? ' data-min-rows="' . $args['min_rows'] . '"' : ''; ?>><?php echo esc_attr( $args['std'] ); ?></textarea>

		<?php 
		
		/**
		 * This allows more granular inclusion of additional HTML immediately after the actual "input"
		 * 
		 * @since		1.2.0
		 * @return		void
		 */
		do_action( 'ld_slack_after_setting_input_html', $args ); ?>

		<?php if ( ! empty( $args['desc'] ) ) : ?>
			<p class="description">
				<?php echo $args['desc']; ?>
			</p>
		<?php endif;
		
		$html = ob_get_clean();
		
		echo apply_filters( 'ld_slack_after_setting_output', $html, $args );

	}
	
}

if ( ! function_exists( 'ld_slack_number_callback' ) ) {

	/**
	 * Creates a Number Field
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_number_callback( $args ) { 

		$args = wp_parse_args( $args, array(
			'id' => '',
			'label' => '',
			'desc' => '',
			'classes' => '',
			'std' => '',
			'placeholder' => '',
			'hidden' => false,
			'min' => 0,
			'max' => '',
			'step' => 1,
			'input_name' => false,
		) );

		if ( ! is_array( $args['classes'] ) ) $args['classes'] = array( $args['classes'] );

		$name = ( $args['input_name'] ) ? $args['input_name'] . '[' . $args['id'] . ']' : $args['id'];
		
		ob_start(); ?>

		<?php if ( ! empty( $args['label'] ) ) : ?>
			<label for="<?php echo esc_attr( $name ); ?>"><?php echo $args['label']; ?></label>
		<?php endif; ?>

		<?php echo do_action( 'ld_slack_after_setting_label', $args ); ?>

		<input type="number" min="<?php echo $args['min']; ?>" max="<?php echo $args['max']; ?>" step="<?php echo $args['step']; ?>" name="<?php echo esc_attr( $name ); ?>" class="ld-slack-field <?php echo ld_slack_sanitize_html_class( $args['classes'] ); ?>" value="<?php echo esc_attr( $args['std'] ); ?>" placeholder="<?php echo esc_attr( $args['placeholder'] ); ?>" />

		<?php 
		
		/**
		 * This allows more granular inclusion of additional HTML immediately after the actual "input"
		 * 
		 * @since		1.2.0
		 * @return		void
		 */
		do_action( 'ld_slack_after_setting_input_html', $args ); ?>

		<?php if ( ! empty( $args['desc'] ) ) : ?>
			<p class="description">
				<?php echo $args['desc']; ?>
			</p>
		<?php endif;
		
		$html = ob_get_clean();
		
		echo apply_filters( 'ld_slack_after_setting_output', $html, $args );

	}
	
}

if ( ! function_exists( 'ld_slack_select_callback' ) ) {

	/**
	 * Creates a Select Field
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_select_callback( $args ) {

		$args = wp_parse_args( $args, array(
			'id' => '',
			'label' => '',
			'desc' => '',
			'classes' => '',
			'std' => '',
			'hidden' => false,
			'input_name' => false,
			'chosen' => false,
			'multiple' => false,
			'placeholder' => '', // Used only for Chosen Multiple
			'choices' => array( 
				0 => '$args[\'choices\'] is empty',
			),
		) );
		
		// Properly set default fallback if the Select Field allows Multiple values
		if ( empty( $args['multiple'] ) ) {
			$args['std'] = isset( $args['std'] ) ? $args['std'] : '';
		} else {
			$args['std'] = ! empty( $args['std'] ) ? $args['std'] : array();
		}

		if ( ! is_array( $args['classes'] ) ) $args['classes'] = array( $args['classes'] );
		
		if ( $args['chosen'] ) {
			$args['classes'][] = 'chosen-select';
		}
		
		if ( $args['multiple'] ) {
			$args['classes'][] = 'ld-slack-select-multiple';
		}

		$name = ( $args['input_name'] ) ? $args['input_name'] . '[' . $args['id'] . ']' : $args['id'];
		$name = ( $args['multiple'] ) ? $name . '[]' : $name;
		
		ob_start(); ?>

		<?php if ( ! empty( $args['label'] ) ) : ?>
			<label for="<?php echo esc_attr( $name ); ?>"><?php echo $args['label']; ?></label>
		<?php endif; ?>

		<?php echo do_action( 'ld_slack_after_setting_label', $args ); ?>

		<select name="<?php echo esc_attr( $name ); ?>" class="ld-slack-field <?php echo ld_slack_sanitize_html_class( $args['classes'] ); ?>"<?php echo ( $args['multiple'] ) ? ' multiple="true"' : ''; ?><?php echo ( $args['multiple'] && $args['placeholder'] ) ? ' data-placeholder="' . $args['placeholder'] . '"' : ''; ?>>
			
			<?php foreach ( $args['choices'] as $key => $value ) : ?>
			
				<?php echo ld_slack_select_innherhtml( $value, $key, $args ); ?>
			
			<?php endforeach; ?>
			
		</select>

		<?php 
		
		/**
		 * This allows more granular inclusion of additional HTML immediately after the actual "input"
		 * 
		 * @since		1.2.0
		 * @return		void
		 */
		do_action( 'ld_slack_after_setting_input_html', $args ); ?>

		<?php if ( ! empty( $args['desc'] ) ) : ?>
			<p class="description">
				<?php echo $args['desc']; ?>
			</p>
		<?php endif;
		
		$html = ob_get_clean();
		
		echo apply_filters( 'ld_slack_after_setting_output', $html, $args );

	}
	
}

if ( ! function_exists( 'ld_slack_select_innerhtml' ) ) {
	
	/**
	 * Builds out <option>s for our <select> Callback recursively
	 * 
	 * @param		string|array $text Either the Text to display for the <option> or an Array of <optgroup> and <option>s
	 * @param		integer      $key  The value of the <option>
	 * @param		array        $args Settings $args
	 *                                     
	 * @since		1.2.0
	 * @return		string       HTML
	 */
	function ld_slack_select_innherhtml( $text, $key, $args ) {
		
		ob_start();
		
		if ( is_array( $text ) ) {
			
			$output = '';
			foreach ( $text['options'] as $option_value => $option_text ) {
				
				$option_value = $key . '-' . $option_value;

				// This will technically grab nested optgroups too. Most browsers just don't handle that well.
				$output .= ld_slack_select_innherhtml( $option_text, $option_value, $args );

			}
			
			?>

			<optgroup label="<?php echo $text['text']; ?>">
				<?php echo $output; ?>
			</optgroup>

			<?php

		}
		else {
			
			if ( ! $args['multiple'] ) : 
		
				$selected = selected( $key, $args['std'], false ); ?>

				<option value="<?php echo esc_attr( $key ); ?>" <?php echo $selected;?>>
					<?php echo esc_html( $text ); ?>
				</option>

			<?php else : 

				// Do an in_array() check to output selected attribute for Multiple ?>
				<option value="<?php echo esc_attr( $key ); ?>"<?php echo ( in_array( (string) $key, $args['std'] ) ) ? ' selected="true"' : ''; ?>>
					<?php echo esc_html( $text ); ?>
				</option>

			<?php endif;

		}
		
		return ob_get_clean();
		
	}
	
}

if ( ! function_exists( 'ld_slack_color_callback' ) ) {

	/**
	 * Creates a Color Field
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_color_callback( $args ) {

		$args = wp_parse_args( $args, array(
			'id' => '',
			'label' => '',
			'desc' => '',
			'classes' => '',
			'std' => '#3299BB',
			'default' => '#3299BB',
			'input_name' => false,
		) );

		if ( ! is_array( $args['classes'] ) ) $args['classes'] = array( $args['classes'] );

		$name = ( $args['input_name'] ) ? $args['input_name'] . '[' . $args['id'] . ']' : $args['id'];
		
		ob_start(); ?>

		<?php if ( ! empty( $args['label'] ) ) : ?>
			<label for="<?php echo esc_attr( $name ); ?>"><?php echo $args['label']; ?></label>
		<?php endif; ?>

		<?php echo do_action( 'ld_slack_after_setting_label', $args ); ?>

		<input type="text" name="<?php echo esc_attr( $name ); ?>" class="ld-slack-field wp-colorpicker <?php echo ld_slack_sanitize_html_class( $args['classes'] ); ?>" value="<?php echo esc_attr( $args['std'] ); ?>" data-default-color="<?php echo esc_attr( $args['default'] ); ?>" />

		<?php 
		
		/**
		 * This allows more granular inclusion of additional HTML immediately after the actual "input"
		 * 
		 * @since		1.2.0
		 * @return		void
		 */
		do_action( 'ld_slack_after_setting_input_html', $args ); ?>
		
		<?php $html = ob_get_clean();
		
		echo apply_filters( 'ld_slack_after_setting_output', $html, $args );

	}
	
}

if ( ! function_exists( 'ld_slack_hook_callback' ) ) {

	/**
	 * Creates a Hook within the Metabox
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_hook_callback( $args ) {

		if ( empty( $args['id'] ) ) {
			echo _x( 'You must enter an ID for this Field Type.', 'LearnDash Slack Hook Field Error', 'learndash-slack' );
			return false;
		}

		do_action( 'ld_slack_' . $args['id'], $args );

	}
	
}

if ( ! function_exists( 'ld_slack_missing_callback' ) ) {
	
	/**
	 * Creates a Missing Callback Message
	 * 
	 * @param		array $args Arguments
	 *						  
	 * @access		public
	 * @since		1.0.0
	 * @return		HTML
	 */
	function ld_slack_missing_callback( $args ) {
	
		printf( 
			_x( 'A callback function called "ld_slack_%s_callback" does not exist.', 'LearnDash Slack Missing Callback', 'learndash-slack' ),
			$args['type']
		);
		
	}
	
}