<?php
/**
 * Integration for LearnDash Gradebook
 *
 * @since 1.2.0
 *
 * @package LD_Slack
 * @subpackage LD_Slack/core/integrations/ld-gradebook
 */

defined( 'ABSPATH' ) || die();

final class LD_Slack_LD_Gradebook {
	
	/**
	 * @param		array Holds all Components across all Gradebooks                                         
	 * @since		1.2.0
	 */ 
	public $components = array();
	
	/**
	 * LD_Slack_LD_Gradebook constructor.
	 *
	 * @since 1.2.0
	 */
	function __construct() {
		
		// We have to wait until `init` to add this because of how LD GB creates its Post Type
		// These only control UI elements, so this is fine. All database interactions work as expected without waiting
		add_action( 'init', function() {
			
			$this->generate_components();
			
			// Add New Comment Trigger
			add_filter( 'ld_slack_triggers', array( $this, 'add_triggers' ) );
			
			// Add new Conditional Fields for the Comment Trigger
			add_filter( 'ld_slack_settings_fields', array( $this, 'add_extra_fields' ) );
			
			// Add our Trigger(s) to the Groups Triggers array
			add_filter( 'ld_slack_groups_triggers', array( $this, 'add_groups_triggers' ) );
			
		} );
		
		// Add code for triggers
		add_action( 'ld_gb_manual_grade_added', array( $this, 'ld_gb_manual_grade_added' ) );
		
		// Inject some Checks before we do Replacements or send the Notification
		add_action( 'ld_slack_before_replacements', array( $this, 'before_notification_replacements' ), 10, 5 );
		
		// Add our own Replacement Strings
		add_filter( 'ld_slack_notifications_replacements', array( $this, 'custom_replacement_strings' ), 10, 5 );
		
		// Add our own Hints for the Replacement Strings
		add_filter( 'ld_slack_text_replacement_hints', array( $this, 'custom_replacement_hints' ), 10, 3 );
		
	}
	
	/**
	 * Add Triggers
	 * 
	 * @param	  array $triggers LearnDash Slack Triggers
	 *										
	 * @access	  public
	 * @since	  1.2.0
	 * @return	  array Modified LearnDash Slack Triggers
	 */
	public function add_triggers( $triggers ) {
		
		if ( empty( $this->get_components() ) ) return $triggers;
		
		$triggers['ld_gb_manual_grade_added'] = __( 'Grade manually added to the Gradebook for a Student', 'learndash-slack' );

		return $triggers;

	}
	
	/**
	 * Fires on manually adding a grade to the Gradebook
	 * 
	 * @param		array $manual_grade Data sent when adding a manual grade
	 *                                                            
	 * @access		public
	 * @since		1.2.0
	 * @return		void
	 */
	public function ld_gb_manual_grade_added( $manual_grade ) {
		
		do_action( 'ld_slack_notify', 'ld_gb_manual_grade_added', array(
			'user_id' => $manual_grade['user_id'],
			'component_id' => $manual_grade['component'],
			'gradebook_id' => $manual_grade['gradebook'],
			'name' => $manual_grade['name'],
			'score_display' => $manual_grade['score_display'],
			'status' => $manual_grade['status'],
		) );
		
	}
	
	/**
	 * Conditionally Showing Fields within the Notification Repeater works by adding the Trigger as a HTML Class Name
	 * 
	 * @param	  array $repeater_fields Notification Repeater Fields
	 *												  
	 * @access	  public
	 * @since	  1.2.0
	 * @return	  array Notification Repeater Fields
	 */
	public function add_extra_fields( $repeater_fields ) {
		
		if ( empty( $this->get_components() ) ) return $repeater_fields;
		
		$index = 0;
		foreach ( $repeater_fields as $key => $value ) {
			
			// Find the Numeric Index of the Before Days Field
			if ( $key == 'before_days' ) {
				break;
			}
			
			$index++;
			
		}
		
		// Create a new Repeater Field
		$components = array(
			'ld_gb_component' => array(
				'type' => 'select',
				'label' => _x( 'Within Gradebook Component', 'Within Gradebook Component Label', 'learndash-slack' ),
				'chosen' => true,
				'multiple' => true,
				'classes' => array(
					'ld-slack-field',
					'ld-slack-ld-gb-component',
					'ld-slack-conditional',
					'pass_quiz',
					'fail_quiz',
					'complete_quiz',
					'upload_assignment',
					'approve_assignment',
					'ld_gb_manual_grade_added',
					'required',
				),
				'std' => '',
				'placeholder' => _x( '-- Select Component --', 'Select Field Default', 'learndash-slack' ),
				'choices' => array( 
					'all' => _x( 'All Components', 'Select Field All', 'learndash-slack' ),
				) + $this->get_components_dropdown(),
			),
			'exclude_ld_gb_component' => array(
				'type' => 'select',
				'label' => __( 'Exclude Gradebook Component (Optional)', 'learndash-slack' ),
				'chosen' => true,
				'multiple' => true,
				'placeholder' => _x( '-- Select Component --', 'Select Field Default', 'learndash-slack' ),
				'classes' => array(
					'ld-slack-exclude-select', // This is a conditional field, but not in the same way as others. It will be hidden/shown based on the value the preceding field
					'ld-slack-exclude-parent-ld-gb-component', // This is used to determine the conditional field parent
				),
				'choices' => $this->get_components_dropdown(),
				'std' => '',
			),
		);
		
		// Insert the new field just before the "Before Days" Field
		LEARNDASHSLACK()->array_insert( $repeater_fields, $index - 1, $components );
		
		return $repeater_fields;
		
	}
	
	/**
	 * Inject some checks on whether or not to bail on the Notification
	 * 
	 * @param		object $post            WP_Post Object for our Saved Notification Data
	 * @param		array  $fields          Fields used to create the Post Meta
	 * @param		string $trigger         Notification Trigger
	 * @param		string $notification_id ID Used for Notification Hooks
	 * @param		array  &$args           $args Array passed from the original Trigger of the process
	 *                                                                                     
	 * @access		public
	 * @since		1.2.0
	 * @return		void
	 */
	public function before_notification_replacements( $post, $fields, $trigger, $notification_id, &$args ) {
		
		if ( $notification_id == 'rbm' ) {
			
			if ( $trigger == 'pass_quiz' ||
				$trigger == 'fail_quiz' ||
			  $trigger == 'complete_quiz' ) {

				// We are allowing empty here since a Quiz-related Notification may not have been saved after 1.2.0 of LD Slack was installed with Gradebook active. Empty is treated as the same as "All". Empty is not possible when saving
				if ( ! empty( $fields['ld_gb_component'] ) &&
					 ! in_array( 'all', $fields['ld_gb_component'] ) ) {
					
					$should_bail = true;
					
					foreach ( $fields['ld_gb_component'] as $gradebook_component ) {
						
						$gradebook_component = explode( '-', $gradebook_component );
						
						$gradebook_id = $gradebook_component[0];
						$component_id = $gradebook_component[1];
						
						$components_array = $this->get_components();
						
						// This should not happen, but may as well check to prevent weird errors
						if ( ! isset( $components_array[ $gradebook_id ] ) || 
							! isset( $components_array[ $gradebook_id ]['options'][ $component_id ] ) ) {
							break;
						}
						
						$component = $components_array[ $gradebook_id ]['options'][ $component_id ];
						
						if ( $component['quizzes_all'] ) {
							$should_bail = false;
							break;
						}

						if ( in_array( $args['quiz_id'], $component['quizzes'] ) ) {
							$should_bail = false;
							break;
						}
						
					}

				}
				else {
					
					$should_bail = false;

					// Support for LD Slack v1.1.X
					if ( ! isset( $fields['exclude_ld_gb_component'] ) ) $fields['exclude_ld_gb_component'] = array();

					foreach ( array_filter( $fields['exclude_ld_gb_component'] ) as $gradebook_component ) {
						
						$gradebook_component = explode( '-', $gradebook_component );
						
						$gradebook_id = $gradebook_component[0];
						$component_id = $gradebook_component[1];
						
						$components_array = $this->get_components();
						
						// This should not happen, but may as well check to prevent weird errors
						if ( ! isset( $components_array[ $gradebook_id ] ) || 
							! isset( $components_array[ $gradebook_id ]['options'][ $component_id ] ) ) {
							$should_bail = true;
							break;
						}
						
						$component = $components_array[ $gradebook_id ]['options'][ $component_id ];
						
						if ( $component['quizzes_all'] ) {
							$should_bail = true;
							break;
						}

						if ( in_array( $args['quiz_id'], $component['quizzes'] ) ) {
							$should_bail = true;
							break;
						}
						
					}
					
				}
				
				// Allows our check to be more accurate
				if ( $should_bail ) {
					$args['bail'] = true;
					return false;
				}
				
			}
			
			if ( $trigger == 'upload_assignment' ||
			  $trigger == 'approve_assignment' ) {

				// We are allowing empty here since a Assignment-related Notification may not have been saved after 1.2.0 of LD Slack was installed with Gradebook active. Empty is treated as the same as "All". Empty is not possible when saving
				if ( ! empty( $fields['ld_gb_component'] ) &&
					 ! in_array( 'all', $fields['ld_gb_component'] ) ) {
					
					$should_bail = true;
					
					foreach ( $fields['ld_gb_component'] as $gradebook_component ) {
						
						$gradebook_component = explode( '-', $gradebook_component );
						
						$gradebook_id = $gradebook_component[0];
						$component_id = $gradebook_component[1];
						
						$components_array = $this->get_components();
						
						// This should not happen, but may as well check to prevent weird errors
						if ( ! isset( $components_array[ $gradebook_id ] ) || 
							! isset( $components_array[ $gradebook_id ]['options'][ $component_id ] ) ) {
							break;
						}
						
						$component = $components_array[ $gradebook_id ]['options'][ $component_id ];
						
						if ( $component['assignments_all'] ) {
							$should_bail = false;
							break;
						}
						
						if ( in_array( $args['lesson_id'], $component['assignment_lessons'] ) ) {
							$should_bail = false;
							break;
						}

						if ( in_array( $args['assignment_id'], $component['assignments'] ) ) {
							$should_bail = false;
							break;
						}
						
					}

				}
				else {
					
					$should_bail = false;

					// Support for LD Slack v1.1.X
					if ( ! isset( $fields['exclude_ld_gb_component'] ) ) $fields['exclude_ld_gb_component'] = array();

					foreach ( array_filter( $fields['exclude_ld_gb_component'] ) as $gradebook_component ) {
						
						$gradebook_component = explode( '-', $gradebook_component );
						
						$gradebook_id = $gradebook_component[0];
						$component_id = $gradebook_component[1];
						
						$components_array = $this->get_components();
						
						// This should not happen, but may as well check to prevent weird errors
						if ( ! isset( $components_array[ $gradebook_id ] ) || 
							! isset( $components_array[ $gradebook_id ]['options'][ $component_id ] ) ) {
							$should_bail = true;
							break;
						}
						
						$component = $components_array[ $gradebook_id ]['options'][ $component_id ];
						
						if ( $component['assignments_all'] ) {
							$should_bail = true;
							break;
						}
						
						if ( in_array( $args['lesson_id'], $component['assignment_lessons'] ) ) {
							$should_bail = true;
							break;
						}

						if ( in_array( $args['assignment_id'], $component['assignments'] ) ) {
							$should_bail = true;
							break;
						}
						
					}
					
				}
				
				// Allows our check to be more accurate
				if ( $should_bail ) {
					$args['bail'] = true;
					return false;
				}
				
			}
			
			if ( $trigger == 'ld_gb_manual_grade_added' ) {
				
				if ( ! is_array( $fields['ld_gb_component'] ) ) $fields['ld_gb_component'] = array( $fields['ld_gb_component'] );
				
				if ( ! in_array( 'all', $fields['ld_gb_component'] ) ) {
					
					$should_bail = true;
					
					foreach( $fields['ld_gb_component'] as $gradebook_component ) {
						
						$gradebook_component = explode( '-', $gradebook_component );
						
						$gradebook_id = $gradebook_component[0];
						$component_id = $gradebook_component[1];
						
						if ( $gradebook_id == $args['gradebook_id'] && 
							$component_id == $args['component_id'] ) {
							$should_bail = false;
							break;
						}
						
					}
					
				}
				else {
					
					$should_bail = false;
					
					if ( ! isset( $fields['exclude_ld_gb_component'] ) ) $fields['exclude_ld_gb_component'] = array();
					
					foreach ( $fields['exclude_ld_gb_component'] as $gradebook_component ) {
						
						$gradebook_component = explode( '-', $gradebook_component );
						
						$gradebook_id = $gradebook_component[0];
						$component_id = $gradebook_component[1];
						
						if ( $gradebook_id == $args['gradebook_id'] && 
							$component_id == $args['component_id'] ) {
							$should_bail = true;
							break;
						}
						
					}
					
				}
				
				// Allows our check to be more accurate
				if ( $should_bail ) {
					$args['bail'] = true;
					return false;
				}
				
				$groups = get_posts( array( 
					'post_type' => 'groups',
					'posts_per_page' => 1,
				) );

				// Don't bail on the notification, but don't run any checks either
				if ( empty( $groups ) ) return false;
				
				if ( ! is_array( $fields['group'] ) ) $fields['group'] = array( $fields['group'] );
				
				// We are allowing empty here since a Group-related Notification may not have been saved after 1.2.0 of LD Slack was installed with Existing Groups. Empty is treated as the same as "All". Empty is not possible when saving
				if ( ! empty( $fields['group'] ) && 
				   ! in_array( 'all', $fields['group'] ) ) {
					
					// Groups the User is in
					$users_groups = learndash_get_users_group_ids( $args['user_id'] );
					
					// If none of the User's Groups exist in our Notification, bail
					if ( empty( array_intersect( $users_groups, $fields['group'] ) ) ) {
						$args['bail'] = true;
						return false;
					}
					
				}
				
			}
			
		}
		
	}
	
	/**
	 * Based on our Notification ID and Trigger, use some extra Replacement Strings
	 * 
	 * @param		array  $replacements    Notification Fields to check for replacements in
	 * @param		array  $fields          Fields used to create the Post Meta
	 * @param		string $trigger         Notification Trigger
	 * @param		string $notification_id ID used for Notification Hooks
	 * @param		array  $args            $args Array passed from the original Trigger of the process
	 *                                                                                     
	 * @access		public
	 * @since		1.2.0
	 * @return		array  Replaced Strings within each Field
	 */
	public function custom_replacement_strings( $replacements, $fields, $trigger, $notification_id, $args ) {

		if ( $notification_id == 'rbm' ) {

			switch ( $trigger ) {

				case 'ld_gb_manual_grade_added':
					
					$grade_statuses = ld_gb_get_grade_statuses();
					
					$replacements['%gradebook_name%'] = get_the_title( $args['gradebook_id'] );
					
					$replacements['%grade_name%'] = $args['name'];
					
					$replacements['%grade_status%'] = ( isset( $grade_statuses[ $args['status'] ]['label'] ) ) ? $grade_statuses[ $args['status'] ]['label'] : __( 'No special status', 'learndash-gradebook' );
					
					if ( ! $args['score_display'] ) {
						$replacements['%grade_score%'] = __( 'No Score Awarded', 'learndash-slack' );
					}
					else {
						$replacements['%grade_score%'] = $args['score_display'];
					}
					
					$is_weighted = ld_gb_get_field( 'gradebook_weighting_enable', $args['gradebook_id'] );
					
					$components = $this->get_components();
					
					// Grab our Component
					$component = ( isset( $components[ $args['gradebook_id'] ] ) && isset( $components[ $args['gradebook_id'] ]['options'][ $args['component_id'] ] ) ) ? $components[ $args['gradebook_id'] ]['options'][ $args['component_id'] ] : array();
					
					if ( empty( $component ) ) {
						
						$replacements['%component_name%'] = __( 'This manually graded item was not assigned a Component', 'learndash-slack' );
						
						if ( $is_weighted ) {
						
							$replacements['%component_weight%'] = __( 'This manually graded item was not assigned a Component', 'learndash-slack' );
								
						}
						
					}
					else {
						
						$replacements['%component_name%'] = $component['name'];
						
						if ( $is_weighted ) {
						
							// LD GB just appends a Percent Sign here. No special methods.
							$replacements['%component_weight%'] = $component['weight'] . '%';
							
						}
						
					}
					
					break;
					
				default:
					break;

			}
			
		}
		
		return $replacements;
		
	}
	
	/**
	 * Add Replacement String Hints
	 * 
	 * @param		array $hints              The main Hints Array
	 * @param		array $user_hints         General Hints for a User. These apply to likely any possible Trigger
	 * @param		array $course_basic_hints The Basic Course Hints. These apply to a lot of different Triggers, so they're available in the Filter easily
	 *                                  
	 * @access		public
	 * @since		1.2.0
	 * @return		array The main Hints Array
	 */
	public function custom_replacement_hints( $hints, $user_hints, $course_basic_hints ) {
		
		$manual_grade_hints = array(
			'%grade_name%' => __( 'The name of this manually graded item.', 'learndash-slack' ),
			'%grade_score%' => __( 'The awarded score of this manually graded item.', 'learndash-slack' ),
			'%grade_status%' => __( 'The status of this manually graded item.', 'learndash-slack' ),
			'%gradebook_name%' => __( 'The Gradebook Name this item was entered into.', 'learndash-slack' ),
			'%component_name%' => __( 'The Gradebook Component Name of this manually graded item.', 'learndash-slack' ),
			'%component_weight%' => __( 'The Gradebook Component Weight of this manually graded item.', 'learndash-slack' ),
		);
		
		$hints['ld_gb_manual_grade_added'] = $user_hints + $manual_grade_hints;
		
		return $hints;
		
	}
	
	/**
	 * Add our Trigger(s) to the Groups Triggers Array
	 * 
	 * @param		array $triggers Groups Triggers
	 *                                
	 * @access		public
	 * @since		1.2.0
	 * @return		array Groups Triggers
	 */
	public function add_groups_triggers( $triggers ) {
		
		$triggers[] = 'ld_gb_manual_grade_added';
		
		return $triggers;
		
	}
	
	/**
	 * Returns the Array of Gradebook Components per Gradebook
	 * 
	 * @access		public
	 * @since		1.2.0
	 * @return		array Gradebook Components
	 */
	public function get_components() {
		return $this->components;
	}
	
	/**
	 * Make minor changes to the Components Array specifically for Dropdowns
	 * There's a lot of information here that is not needed for Dropdowns
	 * 
	 * @access		public
	 * @since		1.2.0
	 * @return		array Gradebook Components
	 */
	public function get_components_dropdown() {
		
		$components_array = $this->get_components();
		
		foreach ( $components_array as $gradebook_id => &$components ) {
			
			foreach ( $components['options'] as $component_id => &$component ) {
				
				$component = $component['name'];
				
			}
			
		}
		
		return $components_array;
		
	}
	
	/**
	 * Builds out the Gradebook Components per Gradebook and stores them in our Object
	 * 
	 * @access		private
	 * @since		1.2.0
	 * @return		void
	 */
	private function generate_components() {
		
		global $wp_query;
		
		$gradebooks = new WP_Query( array(
			'post_type' => 'gradebook',
			'posts_per_page' => -1,
			'post_status' => 'publish',
		) );
		
		if ( $gradebooks->have_posts() ) : 
		
			while ( $gradebooks->have_posts() ) : $gradebooks->the_post();
		
				$components_array = ld_gb_get_field( 'components', get_the_ID() );
		
				if ( ! empty( $components_array ) ) {
					
					foreach ( $components_array as $component ) {
						
						// Ensure we have a place to put our data
						if ( ! isset( $this->components[ get_the_ID() ] ) ) {
							$this->components[ get_the_ID() ] = array(
								'text' => get_the_title(),
								'options' => array(),
							);
						}
						
						$this->components[ get_the_ID() ]['options'][ $component['id'] ] = $component;
						
					}
					
				}
		
			endwhile;
		
			wp_reset_postdata();
		
		endif; 
		
	}
	
}

$integrate = new LD_Slack_LD_Gradebook();