<?php
/**
 * Provides helper functions.
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	die;
}

/**
 * Loads a template. Checks the theme first.
 *
 * @since 1.0.0
 *
 * @param string $template_name The template name.
 * @param array $args Arguments to be extracted into the template.
 */
function ld_convertkit_load_template( $template_name, $args = array() ) {

	extract( $args );

	if ( $template = locate_template( "learndash-convertkit/{$template_name}.php" ) ) {

		include_once $template;

	} else {

		include_once LEARNDASH_CONVERTKIT_DIR . "/core/templates/{$template_name}.php";
	}
}

/**
 * Includes HTML for Subscription Form
 *
 * @since 1.0.0
 *
 * @param integer $course_id Course ID
 */
function ld_convertkit_subscribe_form( $course_id ) {

	if ( ! $course_id ) {

		return false;
	}

	ld_convertkit_load_template( 'subscribe-form', array(
		'course_id' => $course_id,
	) );
}

/**
 * Get an array of subscriber emails from the subscriptions data provided by the API.
 *
 * @since 1.0.0
 *
 * @param array $subscribers Subscriptions data from the API.
 *
 * @return array|bool False on corrupt data, array of emails otherwise.
 */
function ld_convertkit_get_emails_from_subscriptions_data( $subscribers ) {

	$emails = array();

	foreach ( $subscribers as $subcription ) {

		$emails[] = $subcription['subscriber']['email_address'];
	}

	return $emails;
}

/**
 * Searches through an associative array by a specified key.
 *
 * @since 1.0.0
 *
 * @param array $array
 * @param string $key
 * @param mixed $value
 *
 * @return array|bool False if not found, array item otherwise.
 */
function ld_convertkit_search_array_by_key( $array, $key, $value ) {

	foreach ( $array as $item ) {

		if ( $item[ $key ] === $value ) {

			return $item;
		}
	}

	return false;
}

/**
 * Gets the "Subscribed!" text.
 *
 * @since 1.0.0
 *
 * @return string
 */
function ld_convertkit_subscribed_text() {

	/**
	 * Already Subscribed text that shows after a Course Status.
	 *
	 * @since 1.0.0
	 */
	$subscribed_text = apply_filters(
		'learndash_convertkit_already_subscribed_text',
		__( 'You are subscribed to this Course.', 'learndash-convertkit' )
	);

	return $subscribed_text;
}

/**
 * Determines if Course auto sync is enabled.
 *
 * @since 1.0.0
 *
 * @return bool
 */
function ld_convertkit_course_auto_sync_enabled() {

	return ld_convertkit_get_option_field( 'disable_course_sync' ) !== '1';
}

/**
 * Gets a Tag ID from a resource, if it exists.
 *
 * @since 1.0.0
 *
 * @param string|int $resource_id Post ID of resource (Course, Lesson, Tag, Quiz)
 * @param bool $check_exists If true, reaches out to API to see if still exists.
 *
 * @return int Tag ID. 0 if none present.
 */
function ld_convertkit_get_resource_tag( $resource_id, $check_exists = false ) {

	if ( ! LD_ConvertKit_API() ) {

		return 0;
	}

	$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );
	$tag             = (int) get_post_meta( $resource_id, "ld_convertkit_course_tag_{$subscriber_form}", true );

	if ( $check_exists ) {

		$response = LD_ConvertKit_API()->get_tag( $tag );

		if ( $response === 0 ) {

			delete_post_meta( $resource_id, "ld_convertkit_course_tag_{$subscriber_form}" );

			return 0;
		}
	}

	return $tag;
}

/**
 * Gets a Tag ID from a resource, if it exists. If it doesn't exist, create one.
 *
 * @since 1.0.0
 *
 * @param string|int $resource_id Post ID of resource (Course, Lesson, Tag, Quiz)
 * @param array $args Optional args.
 *
 * @return int|bool Tag ID if success. False if error.
 */
function ld_convertkit_ensure_resource_tag( $resource_id, $args = array() ) {

	if ( ! LD_ConvertKit_API() ) {

		return false;
	}

	static $tags;

	$resource_title = get_the_title( $resource_id );

	$args = wp_parse_args( $args, array(
		'name' => $resource_title,
	) );

	$tag = ld_convertkit_get_resource_tag( $resource_id, true );

	if ( $tag !== 0 ) {

		return $tag;
	}

	// Check for tag by name
	if ( $tags === null ) {

		$tags = LD_ConvertKit_API()->get_all_tags();

		$tags = wp_list_pluck( $tags, 'id', 'name' );
	}

	$tag_ID = false;

	if ( isset( $tags[ $resource_title ] ) ) {

		$tag_ID = $tags[ $resource_title ];

	} else {

		// Create tag if not present yet
		$result = LD_ConvertKit_API()->create_tag( $args );

		if ( $result !== 0 ) {

			$tag_ID = $result['id'];
		}
	}

	if ( $tag_ID ) {

		$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );
		update_post_meta( $resource_id, "ld_convertkit_course_tag_{$subscriber_form}", $tag_ID );

		return (int) $tag_ID;

	} else {

		return false;
	}
}

/**
 * Updates the user's convertkit subscriber ID.
 *
 * @since 1.1.0
 *
 * @param string|int $user_ID
 *
 * @return string|int
 */
function ld_convertkit_set_user_subscribe_id( $user_ID, $subscriber_ID ) {

	return update_user_meta( $user_ID, 'ld_convertkit_subscriber_id', $subscriber_ID );
}

/**
 * Gets the user's convertkit subscriber ID.
 *
 * @since 1.1.0
 *
 * @param string|int $user_ID
 *
 * @return string|int
 */
function ld_convertkit_get_user_subscribe_id( $user_ID ) {

	return get_user_meta( $user_ID, 'ld_convertkit_subscriber_id', true );
}

/**
 * Syncs subscribers from ConvertKit down to the site.
 *
 * @since 1.1.0
 *
 * @global WPDB $wpdb
 *
 * @param bool $remove_old If true, will wipe all saved user subscription meta before running.
 *
 * @return bool True on success, false on failure.
 */
function ld_convertkit_sync_subscribers( $remove_old = false ) {

	/** @var WPDB $wpdb */
	global $wpdb;

	if ( ! LD_ConvertKit_API() ) {

		return false;
	}

	$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );

	if ( ! $subscriber_form ) {

		return false;
	}

	// Erase all previous subscription info from users
	$wpdb->query(
		$wpdb->prepare(
			"
			DELETE FROM {$wpdb->usermeta}
			 WHERE meta_key = %s
			",
			'ld_convertkit_subscriber_id'
		)
	);

	$subscribers = LD_ConvertKit_API()->get_form_subscriptions( $subscriber_form );

	foreach ( $subscribers as $subscriber ) {

		$user = get_user_by( 'email', $subscriber['subscriber']['email_address'] );

		if ( ! $user ) {

			continue;
		}

		ld_convertkit_set_user_subscribe_id( $user->ID, $subscriber['subscriber']['id'] );
	}

	return true;
}